<?php

namespace App\Services;

use App\Models\PaymentMethod;
use App\Models\Setting;
use Illuminate\Support\Carbon;

class SmpQrisService
{
    public function create(PaymentMethod $paymentMethod, int $amountBase): array
    {
        $settings = Setting::query()->pluck('value', 'key')->all();

        $qris = $this->normalizeSettingString((string) ($settings['smp.qris_string'] ?? ''));
        if ($qris === '') {
            throw new \RuntimeException('SMP QRIS string belum diisi (setting: smp.qris_string).');
        }

        $expiresMinutes = (int) ($settings['smp.expires_minutes'] ?? 30);
        $uniqueDigits = (int) ($settings['smp.unique_digits'] ?? 2);

        $feeFixed = (int) ($paymentMethod->fee ?? 0);
        $feePercent = (float) ($paymentMethod->fee_percent ?? 0);

        $feePercentAmount = (int) ceil($amountBase * ($feePercent / 100));

        $uniqueCodeMax = max(0, (int) pow(10, max(0, $uniqueDigits)) - 1);
        $uniqueCode = $uniqueCodeMax > 0 ? random_int(1, $uniqueCodeMax) : 0;

        $amountTotal = $amountBase + $feeFixed + $feePercentAmount + $uniqueCode;

        $expiresAt = Carbon::now()->addMinutes(max(1, $expiresMinutes));

        $qrString = $this->generateDynamicQris($qris, $amountTotal);

        return [
            'amount' => $amountTotal,
            'fee' => $feeFixed + $feePercentAmount,
            'unique_code' => $uniqueCode,
            'expires_at' => $expiresAt,
            'expired_time' => $expiresAt->timestamp,
            'payment_name' => $paymentMethod->name ?? 'QRIS',
            'qrString' => $qrString,
        ];
    }

    private function normalizeSettingString(string $value): string
    {
        $v = trim($value);

        if ($v === '') {
            return '';
        }

        $v = stripcslashes($v);
        $v = trim($v);

        if (str_starts_with($v, '"') && str_ends_with($v, '"')) {
            $v = trim($v, '"');
        }

        if (str_starts_with($v, "'") && str_ends_with($v, "'")) {
            $v = trim($v, "'");
        }

        return trim($v);
    }

    private function generateDynamicQris(string $stringQris, int $amount): string
    {
        $qris = str_replace(["\r", "\n", "\t"], '', $stringQris);
        $qris = trim($qris);

        $qris = preg_replace('/6304[0-9A-Fa-f]{4}$/', '', $qris) ?? $qris;

        $qris = str_replace('010211', '010212', $qris);

        $qris = preg_replace('/54\d{2}[0-9.]+/', '', $qris, 1) ?? $qris;

        $amountInt = max(0, (int) $amount);
        $amountStr = sprintf('%06d', $amountInt);
        $amountTag = '54' . '06' . $amountStr;

        $step = explode('5802ID', $qris, 2);
        if (count($step) !== 2) {
            throw new \RuntimeException('QRIS base tidak valid: tag 5802ID tidak ditemukan.');
        }

        $qrisWithAmount = trim($step[0]) . $amountTag . '5802ID' . trim($step[1]);

        $payload = $qrisWithAmount . '6304';

        return $payload . $this->convertCrc16($payload);
    }

    private function convertCrc16(string $str): string
    {
        $crc = 0xFFFF;
        $len = strlen($str);

        for ($c = 0; $c < $len; $c++) {
            $crc ^= (ord($str[$c]) << 8);
            for ($i = 0; $i < 8; $i++) {
                if ($crc & 0x8000) {
                    $crc = (($crc << 1) ^ 0x1021) & 0xFFFF;
                } else {
                    $crc = ($crc << 1) & 0xFFFF;
                }
            }
        }

        return strtoupper(str_pad(dechex($crc), 4, '0', STR_PAD_LEFT));
    }
}