<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class QrispyService
{
    private string $baseUrl;
    private string $apiToken;

    public function __construct()
    {
        $this->baseUrl = 'https://api.qrispy.id';

        $settings = Setting::query()->pluck('value', 'key')->all();
        $rawToken = (string) ($settings['qrispy.api_token'] ?? '');

        $this->apiToken = $this->normalizeSecret($rawToken);
    }

    private function normalizeSecret(string $value): string
    {
        $v = trim($value);

        if ($v === '') {
            return '';
        }

        $v = stripcslashes($v);
        $v = trim($v);

        if (str_starts_with($v, '"') && str_ends_with($v, '"')) {
            $v = trim($v, '"');
        }

        if (str_starts_with($v, "'") && str_ends_with($v, "'")) {
            $v = trim($v, "'");
        }

        return trim($v);
    }

    private function ensureToken(): void
    {
        if ($this->apiToken === '') {
            throw new \RuntimeException('QRISPY API token belum dikonfigurasi.');
        }
    }

    public function generate(int $amount, string $paymentReference): array
    {
        $this->ensureToken();

        if ($amount <= 0) {
            throw new \InvalidArgumentException('Amount tidak valid.');
        }

        $paymentReference = trim($paymentReference);

        if ($paymentReference === '') {
            throw new \InvalidArgumentException('Payment reference tidak valid.');
        }

        $response = Http::timeout(15)
            ->asJson()
            ->withHeaders([
                'X-API-Token' => $this->apiToken,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ])
            ->post($this->baseUrl . '/api/payment/qris/generate', [
                'amount' => $amount,
                'payment_reference' => $paymentReference,
            ]);

        $json = $response->json();

        if (!$response->successful()) {
            Log::warning('[QRISPY] GENERATE_FAILED', [
                'status' => $response->status(),
                'order_ref' => $paymentReference,
                'amount' => $amount,
                'body' => mb_substr((string) $response->body(), 0, 2000),
            ]);

            throw new \RuntimeException('Gagal membuat QRIS QRISPY.');
        }

        if (!is_array($json) || ($json['status'] ?? null) !== 'success') {
            Log::warning('[QRISPY] GENERATE_INVALID_RESPONSE', [
                'order_ref' => $paymentReference,
                'amount' => $amount,
                'body' => mb_substr((string) $response->body(), 0, 2000),
            ]);

            throw new \RuntimeException('Response QRISPY tidak valid.');
        }

        $data = $json['data'] ?? null;

        if (!is_array($data)) {
            Log::warning('[QRISPY] GENERATE_NO_DATA', [
                'order_ref' => $paymentReference,
                'amount' => $amount,
                'body' => mb_substr((string) $response->body(), 0, 2000),
            ]);

            throw new \RuntimeException('Response QRISPY tidak berisi data.');
        }

        return $data;
    }

    public function status(string $qrisId): array
    {
        $this->ensureToken();

        $qrisId = trim($qrisId);

        if ($qrisId === '') {
            throw new \InvalidArgumentException('QRIS ID tidak valid.');
        }

        $response = Http::timeout(15)
            ->asJson()
            ->withHeaders([
                'X-API-Token' => $this->apiToken,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ])
            ->get($this->baseUrl . '/api/payment/qris/' . rawurlencode($qrisId) . '/status');

        $json = $response->json();

        if (!$response->successful()) {
            Log::warning('[QRISPY] STATUS_FAILED', [
                'status' => $response->status(),
                'qris_id' => $qrisId,
                'body' => mb_substr((string) $response->body(), 0, 2000),
            ]);

            throw new \RuntimeException('Gagal mengecek status QRISPY.');
        }

        if (!is_array($json)) {
            Log::warning('[QRISPY] STATUS_INVALID_RESPONSE', [
                'qris_id' => $qrisId,
                'body' => mb_substr((string) $response->body(), 0, 2000),
            ]);

            throw new \RuntimeException('Response QRISPY tidak valid.');
        }

        return $json;
    }
}
