<?php

namespace App\Http\Controllers;

use App\Events\PaymentStatusUpdated;
use App\Events\RealtimeTransaction;
use App\Models\Account;
use App\Models\Order;
use App\Models\Product;
use App\Models\Setting;
use App\Services\DigiflazzService;
use App\Services\WhatsAppNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class TripayCallbackController extends Controller
{
    private ?string $privateKey = null;

    private array $allowedIps = [
        '95.111.200.230',
        '2a04:3543:1000:2310:ac92:4cff:fe87:63f9',
        '127.0.0.1',
    ];

    public function __construct()
    {
        $settings = Setting::query()->pluck('value', 'key')->all();
        $this->privateKey = $this->normalizeSettingString((string) ($settings['tripay.privatekey'] ?? ''));
    }

    public function handle(Request $request)
    {
        if (! $this->isValidIp($request->ip())) {
            Log::warning('Unauthorized Tripay callback attempt from IP: ' . $request->ip());
            return response()->json(['success' => false, 'message' => 'Unauthorized IP'], 403);
        }

        $callbackSignature = (string) $request->header('X-Callback-Signature');

        $json = json_encode(json_decode($request->getContent(), true), JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $signature = hash_hmac('sha256', $json, trim((string) $this->privateKey));

        if ($signature !== $callbackSignature) {
            return response()->json(['success' => false, 'message' => 'Invalid signature'], 401);
        }

        if ($request->header('X-Callback-Event') !== 'payment_status') {
            return response()->json(['success' => false, 'message' => 'Unrecognized callback event'], 400);
        }

        $data = json_decode($json, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return response()->json(['success' => false, 'message' => 'Invalid JSON'], 400);
        }

        if (! isset($data['is_closed_payment']) || (int) $data['is_closed_payment'] !== 1) {
            return response()->json(['success' => false, 'message' => 'Payment not closed'], 400);
        }

        $merchantRef = $data['merchant_ref'] ?? null;
        $status = strtoupper((string) ($data['status'] ?? ''));

        if (! $merchantRef || ! $status) {
            return response()->json(['success' => false, 'message' => 'Invalid callback data'], 400);
        }

        $order = Order::query()
            ->where('order_id', $merchantRef)
            ->where('payment_status', 'UNPAID')
            ->first();

        if (! $order) {
            return response()->json(['success' => false, 'message' => "No order found: $merchantRef"], 404);
        }

        if ($status === 'PAID') {
            $order->update([
                'payment_status' => 'PAID',
                'buy_status' => 'Proses',
            ]);

            if ($order->type === 'Akun') {
                $this->processAccountTransaction($order);
            } else {
                $this->processDigiflazzTransaction($order);
            }

            $order->refresh();

            event(new PaymentStatusUpdated($order));
            event(new RealtimeTransaction($order));

            $this->notify($order, 'payment_paid', [
                'payment_status' => 'PAID',
                'buy_status' => $order->buy_status,
            ]);

            if ($order->buy_status === 'Sukses') {
                $this->notify($order, 'delivered', [
                    'status' => $order->buy_status,
                    'buy_status' => $order->buy_status,
                    'serial_number' => $order->serial_number,
                ]);
            } elseif ($order->buy_status === 'Gagal') {
                $this->notify($order, 'buy_status', [
                    'status' => $order->buy_status,
                    'buy_status' => $order->buy_status,
                    'serial_number' => $order->serial_number,
                ]);
            }

            return response()->json(['success' => true]);
        }

        if ($status === 'EXPIRED') {
            $order->update([
                'payment_status' => 'EXPIRED',
                'buy_status' => 'Batal',
            ]);

            event(new PaymentStatusUpdated($order));
            event(new RealtimeTransaction($order));

            $this->notify($order, 'payment_expired', [
                'payment_status' => 'EXPIRED',
                'buy_status' => $order->buy_status,
            ]);

            return response()->json(['success' => true]);
        }

        if ($status === 'FAILED') {
            $order->update([
                'payment_status' => 'FAILED',
                'buy_status' => 'Batal',
            ]);

            event(new PaymentStatusUpdated($order));
            event(new RealtimeTransaction($order));

            $this->notify($order, 'payment_failed', [
                'payment_status' => 'FAILED',
                'buy_status' => $order->buy_status,
            ]);

            return response()->json(['success' => true]);
        }

        return response()->json(['success' => false, 'message' => 'Unknown status'], 400);
    }

    private function processDigiflazzTransaction(Order $order): void
    {
        app(DigiflazzService::class)->submit($order);
    }

    private function processAccountTransaction(Order $order): void
    {
        $product = Product::where('code', $order->code_product)->first();

        if (! $product) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Product not found',
            ]);

            return;
        }

        if ($product->type_transaction === 'manual') {
            $manualMessage = "Pesanan Anda sedang diproses secara manual.\nAdmin akan segera menghubungi Anda.\nHarap tunggu dan pastikan nomor WhatsApp Anda aktif.";

            $order->update([
                'serial_number' => $manualMessage,
                'buy_status' => 'Proses',
                'df_message' => 'Order will be processed manually',
            ]);

            return;
        }

        $account = Account::where('code_account', $order->code_product)
            ->where('stok', '>', 0)
            ->where('status', 'available')
            ->first();

        if (! $account) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Account not available',
                'serial_number' => 'Stok akun tidak tersedia, hubungi admin untuk mendapatkan bantuan!',
            ]);

            return;
        }

        $credentials = "Berikut detail akun Anda:\nEmail: {$account->email}\nPassword: {$account->password}\n";

        if ($account->descriptions) {
            $credentials .= "Keterangan: {$account->descriptions}\n";
        }

        $order->update([
            'serial_number' => $credentials,
            'buy_status' => 'Sukses',
            'df_message' => 'Account delivered successfully',
        ]);

        $account->decrement('stok');

        if ($account->stok <= 0) {
            $account->update(['status' => 'sold']);
        }
    }

    private function notify(Order $order, string $templateKey, array $extra = []): void
    {
        $vars = array_merge($this->baseVars($order), $extra);

        app(WhatsAppNotificationService::class)->send(
            $order->whatsapp,
            $templateKey,
            $vars
        );
    }

    private function baseVars(Order $order): array
    {
        return [
            'order_id' => $order->order_id,
            'games' => $order->games,
            'product' => $order->product,
            'id_games' => $order->id_games,
            'server_games' => $order->server_games,
            'nickname' => $order->nickname,
            'price' => $order->price,
            'fee' => $order->fee,
            'total_price' => $order->total_price,
            'payment_name' => $order->payment_name,
            'payment_method' => $order->payment_method,
            'payment_code' => $order->payment_code,
            'payment_status' => $order->payment_status,
            'buy_status' => $order->buy_status,
            'expired_time' => $order->expired_time,
            'invoice_url' => config('app.front_url') . "/invoices/{$order->order_id}",
            'serial_number' => $order->serial_number,
            'df_message' => $order->df_message,
            'app_name' => config('app.name'),
        ];
    }

    private function isValidIp(string $ip): bool
    {
        return in_array($ip, $this->allowedIps, true);
    }

    private function normalizeSettingString(string $value): string
    {
        $v = trim($value);

        if ($v === '') {
            return '';
        }

        $v = stripcslashes($v);
        $v = trim($v);

        if (str_starts_with($v, '"') && str_ends_with($v, '"')) {
            $v = trim($v, '"');
        }

        if (str_starts_with($v, "'") && str_ends_with($v, "'")) {
            $v = trim($v, "'");
        }

        return trim($v);
    }
}