<?php

namespace App\Http\Controllers;

use App\Events\PaymentStatusUpdated;
use App\Events\RealtimeTransaction;
use App\Models\Account;
use App\Models\Order;
use App\Models\Product;
use App\Models\QrisPayment;
use App\Models\Setting;
use App\Models\SmpQrisCallback;
use App\Services\DigiflazzService;
use App\Services\WhatsAppNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SmpQrisCallbackController extends Controller
{
    public function handle(Request $request, string $secret)
    {
        $settings = Setting::all()->pluck('value', 'key')->toArray();

        $expectedSecret = $this->normalizeSettingString((string) ($settings['smp.callback_secret'] ?? ''));
        $expectedUsername = $this->normalizeSettingString((string) ($settings['smp.username'] ?? ''));

        Log::info('[SMP-QRIS] HIT', [
            'ip' => $request->ip(),
            'method' => $request->method(),
            'path' => $request->path(),
            'headers' => [
                'content-type' => $request->header('content-type'),
                'user-agent' => $request->header('user-agent'),
            ],
            'payload' => $request->all(),
        ]);

        if ($expectedSecret === '' || ! hash_equals($expectedSecret, (string) $secret)) {
            Log::warning('[SMP-QRIS] UNAUTHORIZED_SECRET', ['provided' => $secret]);

            return response()->json([
                'responseCode' => '4015200',
                'responseMessage' => 'Unauthorized',
            ], 401);
        }

        $payload = $request->all();

        $usUsername = (string) $request->input('us_username', '');
        $rrn = (string) $request->input('rrn', '');
        $trId = (string) $request->input('tr_id', '');
        $issuer = (string) $request->input('issuer', '');
        $payerName = (string) $request->input('PayerName', '');
        $amountValue = (string) data_get($payload, 'amount.value', '');
        $saldoAkhir = $request->has('saldo_akhir') ? (int) $request->input('saldo_akhir') : null;
        $timestamp = (string) $request->input('timestamp', '');

        if ($expectedUsername !== '' && $usUsername !== '' && $usUsername !== $expectedUsername) {
            Log::warning('[SMP-QRIS] USERNAME_INVALID', [
                'expected' => $expectedUsername,
                'got' => $usUsername,
                'rrn' => $rrn,
            ]);

            return response()->json([
                'responseCode' => '4015200',
                'responseMessage' => 'Username invalid',
            ], 200);
        }

        if ($rrn === '') {
            Log::warning('[SMP-QRIS] RRN_EMPTY', ['payload' => $payload]);

            return response()->json([
                'responseCode' => '2005200',
                'responseMessage' => 'Request has been processed successfully',
            ], 200);
        }

        $amountInt = $this->amountToInt($amountValue);

        Log::info('[SMP-QRIS] PARSED', [
            'rrn' => $rrn,
            'tr_id' => $trId,
            'amount_value' => $amountValue,
            'amount_int' => $amountInt,
            'issuer' => $issuer,
            'payer_name' => $payerName,
            'saldo_akhir' => $saldoAkhir,
            'timestamp' => $timestamp,
        ]);

        SmpQrisCallback::query()->updateOrCreate(
            ['rrn' => $rrn],
            [
                'us_username' => $usUsername !== '' ? $usUsername : null,
                'tr_id' => $trId !== '' ? $trId : null,
                'issuer' => $issuer !== '' ? $issuer : null,
                'payer_name' => $payerName !== '' ? $payerName : null,
                'amount_value' => $amountValue !== '' ? $amountValue : null,
                'saldo_akhir' => $saldoAkhir,
                'timestamp' => $timestamp !== '' ? $timestamp : null,
                'payload' => $payload,
            ]
        );

        try {
            $processedOrderId = null;

            $response = DB::transaction(function () use (
                $rrn,
                $trId,
                $issuer,
                $payerName,
                $amountValue,
                $amountInt,
                $saldoAkhir,
                $timestamp,
                &$processedOrderId
            ) {
                $already = QrisPayment::query()
                    ->where('meta', 'like', '%"rrn":"' . $rrn . '"%')
                    ->exists();

                if ($already) {
                    Log::info('[SMP-QRIS] RRN_ALREADY_PROCESSED', ['rrn' => $rrn]);

                    return response()->json([
                        'responseCode' => '2005201',
                        'responseMessage' => 'RRN already processed',
                    ], 200);
                }

                $qrisPayment = QrisPayment::query()
                    ->where('status', 'pending')
                    ->where('amount_total', $amountInt)
                    ->where('expires_at', '>', now())
                    ->orderByDesc('id')
                    ->first();

                if (! $qrisPayment) {
                    Log::warning('[SMP-QRIS] PAYMENT_NOT_FOUND', [
                        'rrn' => $rrn,
                        'amount_int' => $amountInt,
                    ]);

                    SmpQrisCallback::query()->where('rrn', $rrn)->update([
                        'matched_order_id' => null,
                        'qris_payment_id' => null,
                    ]);

                    return response()->json([
                        'responseCode' => '2005200',
                        'responseMessage' => 'Request has been processed successfully',
                    ], 200);
                }

                $order = Order::query()->where('order_id', $qrisPayment->order_id)->first();

                if (! $order) {
                    SmpQrisCallback::query()->where('rrn', $rrn)->update([
                        'matched_order_id' => null,
                        'qris_payment_id' => $qrisPayment->id,
                    ]);

                    Log::warning('[SMP-QRIS] ORDER_NOT_FOUND', [
                        'rrn' => $rrn,
                        'qris_payment_id' => $qrisPayment->id,
                        'order_id' => $qrisPayment->order_id,
                    ]);

                    $qrisPayment->update([
                        'meta' => array_merge($qrisPayment->meta ?? [], [
                            'provider' => 'smp',
                            'rrn' => $rrn,
                            'tr_id' => $trId !== '' ? $trId : null,
                            'issuer' => $issuer !== '' ? $issuer : null,
                            'payer_name' => $payerName !== '' ? $payerName : null,
                            'saldo_akhir' => $saldoAkhir,
                            'amount_value' => $amountValue,
                            'timestamp' => $timestamp !== '' ? $timestamp : null,
                            'note' => 'order_not_found',
                        ]),
                    ]);

                    return response()->json([
                        'responseCode' => '2005200',
                        'responseMessage' => 'Request has been processed successfully',
                    ], 200);
                }

                if ((int) $order->total_price !== (int) $amountInt) {
                    Log::warning('[SMP-QRIS] AMOUNT_MISMATCH', [
                        'rrn' => $rrn,
                        'order_id' => $order->order_id,
                        'order_total_price' => (int) $order->total_price,
                        'callback_amount_int' => $amountInt,
                    ]);

                    $qrisPayment->update([
                        'meta' => array_merge($qrisPayment->meta ?? [], [
                            'provider' => 'smp',
                            'rrn' => $rrn,
                            'tr_id' => $trId !== '' ? $trId : null,
                            'issuer' => $issuer !== '' ? $issuer : null,
                            'payer_name' => $payerName !== '' ? $payerName : null,
                            'saldo_akhir' => $saldoAkhir,
                            'amount_value' => $amountValue,
                            'timestamp' => $timestamp !== '' ? $timestamp : null,
                            'note' => 'amount_mismatch',
                        ]),
                    ]);

                    return response()->json([
                        'responseCode' => '2005200',
                        'responseMessage' => 'Request has been processed successfully',
                    ], 200);
                }

                $qrisPayment->update([
                    'status' => 'paid',
                    'meta' => array_merge($qrisPayment->meta ?? [], [
                        'provider' => 'smp',
                        'rrn' => $rrn,
                        'tr_id' => $trId !== '' ? $trId : null,
                        'issuer' => $issuer !== '' ? $issuer : null,
                        'payer_name' => $payerName !== '' ? $payerName : null,
                        'saldo_akhir' => $saldoAkhir,
                        'amount_value' => $amountValue,
                        'timestamp' => $timestamp !== '' ? $timestamp : null,
                    ]),
                ]);

                SmpQrisCallback::query()->where('rrn', $rrn)->update([
                    'matched_order_id' => $order->order_id,
                    'qris_payment_id' => $qrisPayment->id,
                ]);

                $wasUnpaid = $order->payment_status === 'UNPAID';

                if ($wasUnpaid) {
                    $order->update([
                        'payment_status' => 'PAID',
                        'buy_status' => 'Proses',
                    ]);

                    $processedOrderId = $order->order_id;
                }

                Log::info('[SMP-QRIS] PAID_OK', [
                    'rrn' => $rrn,
                    'order_id' => $order->order_id,
                    'qris_payment_id' => $qrisPayment->id,
                    'amount' => $amountInt,
                    'trigger_process' => (bool) $processedOrderId,
                ]);

                return response()->json([
                    'responseCode' => '2005200',
                    'responseMessage' => 'Request has been processed successfully',
                ], 200);
            });

            if ($processedOrderId) {
                DB::afterCommit(function () use ($processedOrderId) {
                    $order = Order::where('order_id', $processedOrderId)->first();

                    if (! $order) {
                        return;
                    }

                    if ($order->type === 'Akun') {
                        $this->processAccountTransaction($order);
                    } else {
                        $this->processDigiflazzTransaction($order);
                    }

                    event(new PaymentStatusUpdated($order));
                    event(new RealtimeTransaction($order));

                    $this->notify($order, 'payment_paid', [
                        'payment_status' => 'PAID',
                        'buy_status' => $order->buy_status,
                    ]);
                });
            }

            return $response;
        } catch (\Throwable $e) {
            Log::error('[SMP-QRIS] ERROR', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);

            return response()->json([
                'responseCode' => '2005200',
                'responseMessage' => 'Request has been processed successfully',
            ], 200);
        }
    }

    private function processDigiflazzTransaction(Order $order): void
    {
        $result = app(DigiflazzService::class)->submit($order);
        $order->refresh();

        if (($result['success'] ?? false) && $order->buy_status === 'Sukses') {
            $this->notify($order, 'delivered', [
                'status' => $order->buy_status,
                'buy_status' => $order->buy_status,
                'serial_number' => $order->serial_number,
                'df_message' => $order->df_message,
            ]);

            return;
        }

        if ($order->buy_status === 'Gagal') {
            $this->notify($order, 'buy_status', [
                'status' => $order->buy_status,
                'buy_status' => $order->buy_status,
                'df_message' => $order->df_message,
            ]);
        }
    }

    private function processAccountTransaction(Order $order): void
    {
        $product = Product::where('code', $order->code_product)->first();

        if (! $product) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Product not found',
            ]);

            $this->notify($order, 'buy_status', [
                'status' => $order->buy_status,
                'buy_status' => $order->buy_status,
            ]);

            return;
        }

        if ($product->type_transaction === 'manual') {
            $manualMessage = "Pesanan Anda sedang diproses secara manual.\nAdmin akan segera menghubungi Anda.\nHarap tunggu dan pastikan nomor WhatsApp Anda aktif.";

            $order->update([
                'serial_number' => $manualMessage,
                'buy_status' => 'Proses',
                'df_message' => 'Order will be processed manually',
            ]);

            $this->notify($order, 'buy_status', [
                'status' => $order->buy_status,
                'buy_status' => $order->buy_status,
                'serial_number' => $order->serial_number,
            ]);

            return;
        }

        $account = Account::where('code_account', $order->code_product)
            ->where('stok', '>', 0)
            ->where('status', 'available')
            ->first();

        if (! $account) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Account not available',
                'serial_number' => 'Stok akun tidak tersedia, hubungi admin untuk mendapatkan bantuan!',
            ]);

            $this->notify($order, 'buy_status', [
                'status' => $order->buy_status,
                'buy_status' => $order->buy_status,
                'serial_number' => $order->serial_number,
            ]);

            return;
        }

        $credentials = "Berikut detail akun Anda:\nEmail: {$account->email}\nPassword: {$account->password}\n";

        if ($account->descriptions) {
            $credentials .= "Keterangan: {$account->descriptions}\n";
        }

        $order->update([
            'serial_number' => $credentials,
            'buy_status' => 'Sukses',
            'df_message' => 'Account delivered successfully',
        ]);

        $account->decrement('stok');

        if ($account->stok <= 0) {
            $account->update(['status' => 'sold']);
        }

        $this->notify($order, 'delivered', [
            'status' => $order->buy_status,
            'buy_status' => $order->buy_status,
            'serial_number' => $order->serial_number,
        ]);
    }

    private function notify(Order $order, string $templateKey, array $extra = []): void
    {
        $vars = array_merge($this->baseVars($order), $extra);

        app(WhatsAppNotificationService::class)->send(
            $order->whatsapp,
            $templateKey,
            $vars
        );
    }

    private function baseVars(Order $order): array
    {
        return [
            'order_id' => $order->order_id,
            'games' => $order->games,
            'product' => $order->product,
            'id_games' => $order->id_games,
            'server_games' => $order->server_games,
            'nickname' => $order->nickname,
            'price' => $order->price,
            'fee' => $order->fee,
            'total_price' => $order->total_price,
            'payment_name' => $order->payment_name,
            'payment_method' => $order->payment_method,
            'payment_code' => $order->payment_code,
            'payment_status' => $order->payment_status,
            'buy_status' => $order->buy_status,
            'expired_time' => $order->expired_time,
            'invoice_url' => config('app.front_url') . "/invoices/{$order->order_id}",
            'serial_number' => $order->serial_number,
            'df_message' => $order->df_message,
            'app_name' => config('app.name'),
        ];
    }

    private function normalizeSettingString(string $value): string
    {
        $v = trim($value);

        if ($v === '') {
            return '';
        }

        $v = stripcslashes($v);
        $v = trim($v);

        if (str_starts_with($v, '"') && str_ends_with($v, '"')) {
            $v = trim($v, '"');
        }

        if (str_starts_with($v, "'") && str_ends_with($v, "'")) {
            $v = trim($v, "'");
        }

        return trim($v);
    }

    private function amountToInt(string $value): int
{
    $v = trim($value);

    if ($v === '') {
        return 0;
    }

    $v = str_replace(["\xc2\xa0", ' '], '', $v);
    $v = preg_replace('/[^\d.,]/', '', $v) ?? '';

    if ($v === '') {
        return 0;
    }

    $lastDot = strrpos($v, '.');
    $lastComma = strrpos($v, ',');

    $sepPos = null;

    if ($lastDot !== false || $lastComma !== false) {
        if ($lastDot === false) {
            $sepPos = $lastComma;
        } elseif ($lastComma === false) {
            $sepPos = $lastDot;
        } else {
            $sepPos = max($lastDot, $lastComma);
        }
    }

    if ($sepPos !== null) {
        $right = substr($v, $sepPos + 1);
        $left = substr($v, 0, $sepPos);

        if (ctype_digit($right) && strlen($right) === 2) {
            $leftDigits = preg_replace('/\D/', '', $left) ?? '';
            return $leftDigits === '' ? 0 : (int) $leftDigits;
        }
    }

    $digits = preg_replace('/\D/', '', $v) ?? '';
    return $digits === '' ? 0 : (int) $digits;
}
}