<?php

namespace App\Http\Controllers;

use App\Events\PaymentStatusUpdated;
use App\Events\RealtimeTransaction;
use App\Models\Account;
use App\Models\Order;
use App\Models\Product;
use App\Models\Setting;
use App\Services\DigiflazzService;
use App\Services\WhatsAppNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class QrispyWebhookController extends Controller
{
    private string $webhookSecret;
    private string $referencePrefix;

    public function __construct()
    {
        $settings = Setting::query()->pluck('value', 'key')->all();

        $this->webhookSecret = $this->normalizeSecret((string) ($settings['qrispy.webhook_secret'] ?? ''));
        $this->referencePrefix = trim((string) ($settings['qrispy.reference_prefix'] ?? 'Order-'));
        if ($this->referencePrefix === '') {
            $this->referencePrefix = 'Order-';
        }
    }

    private function normalizeSecret(string $value): string
    {
        $v = trim($value);
        if ($v === '') return '';

        $v = stripcslashes($v);
        $v = trim($v);

        if (str_starts_with($v, '"') && str_ends_with($v, '"')) {
            $v = trim($v, '"');
        }

        if (str_starts_with($v, "'") && str_ends_with($v, "'")) {
            $v = trim($v, "'");
        }

        return trim($v);
    }

    public function handle(Request $request)
    {
        if ($this->webhookSecret === '') {
            return response()->json(['success' => false, 'message' => 'Webhook secret not configured'], 500);
        }

        $payload = (string) $request->getContent();
        $signature = (string) $request->header('X-Qrispy-Signature');

        if ($payload === '' || $signature === '') {
            return response()->json(['success' => false, 'message' => 'Invalid request'], 400);
        }

        $expected = hash_hmac('sha256', $payload, $this->webhookSecret);

        if (!hash_equals($expected, $signature)) {
            return response()->json(['success' => false, 'message' => 'Invalid signature'], 401);
        }

        $json = json_decode($payload, true);

        if (!is_array($json)) {
            return response()->json(['success' => false, 'message' => 'Invalid JSON'], 400);
        }

        if (($json['event'] ?? null) !== 'payment.received') {
            return response()->json(['success' => true, 'ignored' => true]);
        }

        $data = $json['data'] ?? null;
        if (!is_array($data)) {
            return response()->json(['success' => false, 'message' => 'Invalid payload'], 400);
        }

        $uniqueId = $data['unique_id'] ?? null;
        $paymentReference = trim((string) ($data['payment_reference'] ?? ''));
        $qrisId = trim((string) ($data['qris_id'] ?? ''));
        $receivedAmount = (int) ($data['received_amount'] ?? 0);
        $paidAtRaw = (string) ($data['paid_at'] ?? '');

        if ($uniqueId === null || $paymentReference === '' || $receivedAmount <= 0) {
            return response()->json(['success' => false, 'message' => 'Missing required fields'], 422);
        }

        $orderId = $paymentReference;
        if (str_starts_with($paymentReference, $this->referencePrefix)) {
            $orderId = substr($paymentReference, strlen($this->referencePrefix));
        }
        $orderId = trim((string) $orderId);

        if ($orderId === '') {
            return response()->json(['success' => false, 'message' => 'Invalid reference'], 422);
        }

        $already = Order::query()
            ->where('qrispy_webhook_unique_id', (string) $uniqueId)
            ->first();

        if ($already) {
            return response()->json(['success' => true]);
        }

        try {
            $order = DB::transaction(function () use ($orderId, $uniqueId, $qrisId, $receivedAmount, $paidAtRaw) {
                $order = Order::query()
                    ->where('order_id', $orderId)
                    ->lockForUpdate()
                    ->first();

                if (!$order) {
                    return null;
                }

                if ($order->qrispy_webhook_unique_id !== null && (string) $order->qrispy_webhook_unique_id === (string) $uniqueId) {
                    return $order;
                }

                if (strtoupper((string) $order->payment_status) === 'PAID') {
                    if ($order->qrispy_webhook_unique_id === null) {
                        $order->qrispy_webhook_unique_id = (string) $uniqueId;
                        $order->save();
                    }
                    return $order;
                }

                $expectedTotal = (int) ($order->total_price ?? 0);
                if ($expectedTotal > 0 && $receivedAmount < $expectedTotal) {
                    return $order;
                }

                $paidAt = null;
                if ($paidAtRaw !== '') {
                    try {
                        $paidAt = Carbon::parse($paidAtRaw);
                    } catch (\Throwable $e) {
                        $paidAt = null;
                    }
                }

                $order->payment_status = 'PAID';
                $order->buy_status = 'Proses';
                $order->qrispy_webhook_unique_id = (string) $uniqueId;

                if ($qrisId !== '' && empty($order->qrispy_qris_id)) {
                    $order->qrispy_qris_id = $qrisId;
                }

                if ($paidAt) {
                    $order->qrispy_paid_at = $paidAt;
                }

                $order->save();

                return $order;
            });

            if (!$order) {
                return response()->json(['success' => false, 'message' => 'Order not found'], 404);
            }

            $order->refresh();

            if (strtoupper((string) $order->payment_status) === 'PAID') {
                if ($order->type === 'Akun') {
                    $this->processAccountTransaction($order);
                } else {
                    $this->processDigiflazzTransaction($order);
                }

                $order->refresh();

                event(new PaymentStatusUpdated($order));
                event(new RealtimeTransaction($order));

                $this->notify($order, 'payment_paid', [
                    'payment_status' => 'PAID',
                    'buy_status' => $order->buy_status,
                ]);

                if ($order->buy_status === 'Sukses') {
                    $this->notify($order, 'delivered', [
                        'status' => $order->buy_status,
                        'buy_status' => $order->buy_status,
                        'serial_number' => $order->serial_number,
                    ]);
                } elseif ($order->buy_status === 'Gagal') {
                    $this->notify($order, 'buy_status', [
                        'status' => $order->buy_status,
                        'buy_status' => $order->buy_status,
                        'serial_number' => $order->serial_number,
                    ]);
                }
            } else {
                Log::warning('[QRISPY] RECEIVED_AMOUNT_TOO_LOW', [
                    'order_id' => $order->order_id,
                    'expected_total' => (int) ($order->total_price ?? 0),
                ]);
            }

            return response()->json(['success' => true]);
        } catch (\Throwable $e) {
            Log::error('[QRISPY] WEBHOOK_FAILED', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);

            return response()->json(['success' => false], 500);
        }
    }

    private function processDigiflazzTransaction(Order $order): void
    {
        app(DigiflazzService::class)->submit($order);
    }

    private function processAccountTransaction(Order $order): void
    {
        $product = Product::where('code', $order->code_product)->first();

        if (!$product) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Product not found',
            ]);
            return;
        }

        if ($product->type_transaction === 'manual') {
            $manualMessage = "Pesanan Anda sedang diproses secara manual.\nAdmin akan segera menghubungi Anda.\nHarap tunggu dan pastikan nomor WhatsApp Anda aktif.";

            $order->update([
                'serial_number' => $manualMessage,
                'buy_status' => 'Proses',
                'df_message' => 'Order will be processed manually',
            ]);
            return;
        }

        $account = Account::where('code_account', $order->code_product)
            ->where('stok', '>', 0)
            ->where('status', 'available')
            ->first();

        if (!$account) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Account not available',
                'serial_number' => 'Stok akun tidak tersedia, hubungi admin untuk mendapatkan bantuan!',
            ]);
            return;
        }

        $credentials = "Berikut detail akun Anda:\nEmail: {$account->email}\nPassword: {$account->password}\n";

        if ($account->descriptions) {
            $credentials .= "Keterangan: {$account->descriptions}\n";
        }

        $order->update([
            'serial_number' => $credentials,
            'buy_status' => 'Sukses',
            'df_message' => 'Account delivered successfully',
        ]);

        $account->decrement('stok');

        if ($account->stok <= 0) {
            $account->update(['status' => 'sold']);
        }
    }

    private function notify(Order $order, string $templateKey, array $extra = []): void
    {
        $vars = array_merge([
            'order_id' => $order->order_id,
            'games' => $order->games,
            'product' => $order->product,
            'nickname' => $order->nickname,
            'id_games' => $order->id_games,
            'server_games' => $order->server_games,
            'price' => (int) ($order->price ?? 0),
            'fee' => (int) ($order->fee ?? 0),
            'total_price' => (int) ($order->total_price ?? 0),
            'payment_name' => $order->payment_name,
            'payment_method' => $order->payment_method,
            'payment_code' => 'Scan QR di halaman invoice',
            'payment_status' => $order->payment_status,
            'buy_status' => $order->buy_status,
            'expired_time' => (int) ($order->expired_time ?? 0),
            'invoice_url' => config('app.front_url') . "/invoices/{$order->order_id}",
            'app_name' => config('app.name'),
            'serial_number' => $order->serial_number,
        ], $extra);

        app(WhatsAppNotificationService::class)->send(
            $order->whatsapp,
            $templateKey,
            $vars,
            null
        );
    }
}
