<?php

namespace App\Http\Controllers\Api;

use App\Events\PaymentStatusUpdated;
use App\Events\RealtimeTransaction;
use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Setting;
use App\Services\DigiflazzService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class QrispyWebhookController extends Controller
{
    public function handle(Request $request)
    {
        $payload = (string) $request->getContent();
        $signature = (string) $request->header('X-Qrispy-Signature', '');
        $ip = (string) $request->ip();

        $body = null;
        try {
            $body = json_decode($payload, true);
        } catch (\Throwable $e) {
            $body = null;
        }

        $event = is_array($body) ? (string) ($body['event'] ?? '') : '';
        $data = (is_array($body) ? ($body['data'] ?? null) : null);
        $dataArr = is_array($data) ? $data : [];

        $qrisId = (string) ($dataArr['qris_id'] ?? '');
        $paymentReference = (string) ($dataArr['payment_reference'] ?? '');
        $uniqueId = $dataArr['unique_id'] ?? null;

        $settings = Setting::query()->pluck('value', 'key')->all();
        $secret = trim((string) ($settings['qrispy.webhook_secret'] ?? ''));

        if ($secret === '') {
            Log::warning('[QRISPY] WEBHOOK_SECRET_EMPTY', [
                'ip' => $ip,
                'event' => $event,
                'qris_id' => $qrisId,
                'payment_reference' => $paymentReference,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Webhook secret belum dikonfigurasi.',
            ], 500);
        }

        $expectedSignature = hash_hmac('sha256', $payload, $secret);
        $verified = ($signature !== '') && hash_equals($expectedSignature, $signature);

        Log::info('[QRISPY] WEBHOOK_HIT', [
            'ip' => $ip,
            'verified' => $verified,
            'event' => $event,
            'qris_id' => $qrisId,
            'payment_reference' => $paymentReference,
            'unique_id' => is_scalar($uniqueId) ? $uniqueId : null,
        ]);

        if (!$verified) {
            Log::warning('[QRISPY] WEBHOOK_INVALID_SIGNATURE', [
                'ip' => $ip,
                'event' => $event,
                'qris_id' => $qrisId,
                'payment_reference' => $paymentReference,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Signature tidak valid.',
            ], 401);
        }

        // Dashboard "Test Webhook" kadang mengirim event selain payment.received
        if ($event !== 'payment.received') {
            Log::info('[QRISPY] WEBHOOK_IGNORED', [
                'event' => $event,
                'qris_id' => $qrisId,
                'payment_reference' => $paymentReference,
            ]);

            return response()->json([
                'success' => true,
                'ignored' => true,
                'event' => $event,
            ], 200);
        }

        if (!is_array($dataArr) || empty($paymentReference)) {
            Log::warning('[QRISPY] WEBHOOK_INVALID_PAYLOAD', [
                'event' => $event,
                'qris_id' => $qrisId,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Payload tidak valid.',
            ], 422);
        }

        $referencePrefix = trim((string) ($settings['qrispy.reference_prefix'] ?? ''));

        $orderId = $paymentReference;
        if ($referencePrefix !== '' && str_starts_with($paymentReference, $referencePrefix)) {
            $orderId = substr($paymentReference, strlen($referencePrefix));
        }
        $orderId = trim((string) $orderId);

        if ($orderId === '') {
            Log::warning('[QRISPY] WEBHOOK_INVALID_REFERENCE', [
                'payment_reference' => $paymentReference,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Payment reference tidak valid.',
            ], 422);
        }

        if ($uniqueId && Order::query()->where('qrispy_webhook_unique_id', $uniqueId)->exists()) {
            Log::info('[QRISPY] WEBHOOK_DUPLICATE', [
                'unique_id' => $uniqueId,
                'order_id' => $orderId,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Duplicate webhook ignored.',
            ], 200);
        }

        $receivedAmount = (int) ($dataArr['received_amount'] ?? 0);
        $paidAtRaw = $dataArr['paid_at'] ?? null;

        $paidAt = null;
        if (!empty($paidAtRaw)) {
            try {
                $paidAt = Carbon::parse((string) $paidAtRaw);
            } catch (\Throwable $e) {
                $paidAt = null;
            }
        }

        $processedOrderId = null;

        $order = DB::transaction(function () use (
            $orderId,
            $qrisId,
            $receivedAmount,
            $paidAt,
            $uniqueId,
            &$processedOrderId
        ) {
            $o = Order::query()->lockForUpdate()->where('order_id', $orderId)->first();

            if (!$o) {
                return null;
            }

            if ($o->payment_status !== 'UNPAID') {
                return $o;
            }

            $totalPrice = (int) ($o->total_price ?? 0);
            if ($receivedAmount > 0 && $totalPrice > 0 && $receivedAmount < $totalPrice) {
                Log::warning('[QRISPY] WEBHOOK_AMOUNT_MISMATCH', [
                    'order_id' => $o->order_id,
                    'received_amount' => $receivedAmount,
                    'total_price' => $totalPrice,
                ]);

                return $o;
            }

            $o->payment_status = 'PAID';
            $o->buy_status = 'Proses';
            $o->qrispy_qris_id = $o->qrispy_qris_id ?: $qrisId;
            $o->qrispy_paid_at = $paidAt ?? now();
            $o->qrispy_received_amount = $receivedAmount ?: null;
            $o->qrispy_webhook_unique_id = $uniqueId ?: null;
            $o->save();

            $processedOrderId = $o->order_id;

            Log::info('[QRISPY] WEBHOOK_MARK_PAID', [
                'order_id' => $o->order_id,
                'qris_id' => $qrisId,
                'received_amount' => $receivedAmount,
            ]);

            return $o;
        });

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order tidak ditemukan.',
            ], 404);
        }

        if ($processedOrderId) {
            DB::afterCommit(function () use ($processedOrderId) {
                $o = Order::query()->where('order_id', $processedOrderId)->first();
                if (!$o) return;

                try {
                    if ($o->type === 'Akun') {
                        // Akun diproses di controller invoice (atau flow existing kamu)
                    } else {
                        app(DigiflazzService::class)->submit($o);
                    }
                } catch (\Throwable $e) {
                    Log::error('[QRISPY] AFTER_COMMIT_PROCESS_FAILED', [
                        'order_id' => $processedOrderId,
                        'message' => $e->getMessage(),
                    ]);
                }

                $o->refresh();

                event(new PaymentStatusUpdated($o));
                event(new RealtimeTransaction($o));
            });
        } else {
            event(new PaymentStatusUpdated($order));
            event(new RealtimeTransaction($order));
        }

        return response()->json([
            'success' => true,
            'message' => 'Webhook processed.',
        ], 200);
    }
}