<?php

namespace App\Http\Controllers\Api;

use App\Events\PaymentStatusUpdated;
use App\Events\RealtimeTransaction;
use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\Order;
use App\Models\Product;
use App\Models\Setting;
use App\Services\DigiflazzService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class QrispyWebhookController extends Controller
{
    private function normalizeSecret(string $value): string
    {
        $v = trim($value);
        if ($v === '') return '';
        $v = stripcslashes($v);
        $v = trim($v);
        if (str_starts_with($v, '"') && str_ends_with($v, '"')) $v = trim($v, '"');
        if (str_starts_with($v, "'") && str_ends_with($v, "'")) $v = trim($v, "'");
        return trim($v);
    }

    public function handle(Request $request)
    {
        $payload = (string) $request->getContent();
        $signature = (string) $request->header('X-Qrispy-Signature');
        $ua = (string) $request->userAgent();
        if (strlen($ua) > 120) $ua = substr($ua, 0, 120);

        Log::info('[QRISPY] WEBHOOK_RECEIVED', [
            'path' => $request->path(),
            'ip' => $request->ip(),
            'len' => strlen($payload),
            'signature_present' => $signature !== '',
            'content_type' => (string) $request->header('Content-Type'),
            'user_agent' => $ua,
        ]);

        $settings = Setting::query()->pluck('value', 'key')->all();

        $secret = $this->normalizeSecret((string) ($settings['qrispy.webhook_secret'] ?? ''));
        if ($secret === '') {
            return response()->json(['success' => false, 'message' => 'Webhook secret not configured'], 500);
        }

        $referencePrefix = trim((string) ($settings['qrispy.reference_prefix'] ?? 'Order-'));
        if ($referencePrefix === '') $referencePrefix = 'Order-';

        if ($payload === '' || $signature === '') {
            Log::warning('[QRISPY] WEBHOOK_REJECTED', [
                'path' => $request->path(),
                'ip' => $request->ip(),
                'reason' => 'missing_payload_or_signature',
            ]);
            return response()->json(['success' => false, 'message' => 'Invalid request'], 400);
        }

        $expected = hash_hmac('sha256', $payload, $secret);

        if (!hash_equals($expected, $signature)) {
            Log::warning('[QRISPY] WEBHOOK_REJECTED', [
                'path' => $request->path(),
                'ip' => $request->ip(),
                'reason' => 'invalid_signature',
            ]);
            return response()->json(['success' => false, 'message' => 'Invalid signature'], 401);
        }

        $json = json_decode($payload, true);
        if (!is_array($json)) {
            Log::warning('[QRISPY] WEBHOOK_REJECTED', [
                'path' => $request->path(),
                'ip' => $request->ip(),
                'reason' => 'invalid_json',
            ]);
            return response()->json(['success' => false, 'message' => 'Invalid JSON'], 400);
        }

        $event = (string) ($json['event'] ?? '');
        Log::info('[QRISPY] WEBHOOK_EVENT', [
            'path' => $request->path(),
            'ip' => $request->ip(),
            'event' => $event,
        ]);

        if ($event !== 'payment.received') {
            Log::info('[QRISPY] WEBHOOK_IGNORED', [
                'path' => $request->path(),
                'ip' => $request->ip(),
                'event' => $event,
            ]);
            return response()->json(['success' => true, 'ignored' => true]);
        }

        $data = $json['data'] ?? null;
        if (!is_array($data)) {
            return response()->json(['success' => false, 'message' => 'Invalid payload'], 422);
        }

        $uniqueId = $data['unique_id'] ?? null;
        $paymentReference = trim((string) ($data['payment_reference'] ?? ''));
        $qrisId = trim((string) ($data['qris_id'] ?? ''));
        $amount = (int) ($data['amount'] ?? 0);
        $receivedAmount = (int) ($data['received_amount'] ?? 0);
        $paidAtRaw = trim((string) ($data['paid_at'] ?? ''));

        if ($uniqueId === null || $paymentReference === '' || $receivedAmount <= 0) {
            Log::warning('[QRISPY] WEBHOOK_REJECTED', [
                'path' => $request->path(),
                'ip' => $request->ip(),
                'reason' => 'missing_required_fields',
            ]);
            return response()->json(['success' => false, 'message' => 'Missing required fields'], 422);
        }

        $orderId = $paymentReference;
        if (str_starts_with($paymentReference, $referencePrefix)) {
            $orderId = substr($paymentReference, strlen($referencePrefix));
        }
        $orderId = trim((string) $orderId);

        if ($orderId === '') {
            Log::warning('[QRISPY] WEBHOOK_REJECTED', [
                'path' => $request->path(),
                'ip' => $request->ip(),
                'reason' => 'invalid_payment_reference',
                'payment_reference' => $paymentReference,
                'unique_id' => (string) $uniqueId,
            ]);
            return response()->json(['success' => false, 'message' => 'Invalid payment_reference'], 422);
        }

        Log::info('[QRISPY] PAYMENT_RECEIVED', [
            'order_id' => $orderId,
            'payment_reference' => $paymentReference,
            'unique_id' => (string) $uniqueId,
            'qris_id' => $qrisId,
            'amount' => $amount,
            'received_amount' => $receivedAmount,
            'paid_at' => $paidAtRaw !== '' ? $paidAtRaw : null,
        ]);

        $already = Order::query()
            ->where('qrispy_webhook_unique_id', (string) $uniqueId)
            ->first();

        if ($already) {
            Log::info('[QRISPY] WEBHOOK_DUPLICATE', [
                'order_id' => $already->order_id,
                'unique_id' => (string) $uniqueId,
            ]);
            return response()->json(['success' => true]);
        }

        try {
            $processedOrderId = null;

            $updatedOrder = DB::transaction(function () use ($orderId, $uniqueId, $qrisId, $amount, $receivedAmount, $paidAtRaw, &$processedOrderId) {
                $locked = Order::query()->lockForUpdate()->where('order_id', $orderId)->first();

                if (!$locked) {
                    return null;
                }

                if ($locked->qrispy_webhook_unique_id !== null && (string) $locked->qrispy_webhook_unique_id === (string) $uniqueId) {
                    return $locked;
                }

                if (strtoupper((string) $locked->payment_status) !== 'UNPAID') {
                    if ($locked->qrispy_webhook_unique_id === null) {
                        $locked->qrispy_webhook_unique_id = (string) $uniqueId;
                        $locked->save();
                    }
                    return $locked;
                }

                $expectedTotal = (int) ($locked->total_price ?? 0);
                if ($expectedTotal > 0 && $receivedAmount < $expectedTotal) {
                    return $locked;
                }

                $paidAt = null;
                if ($paidAtRaw !== '') {
                    try {
                        $paidAt = Carbon::parse($paidAtRaw);
                    } catch (\Throwable $e) {
                        $paidAt = null;
                    }
                }

                $locked->payment_status = 'PAID';
                $locked->buy_status = 'Proses';
                $locked->qrispy_webhook_unique_id = (string) $uniqueId;

                if ($qrisId !== '' && empty($locked->qrispy_qris_id)) {
                    $locked->qrispy_qris_id = $qrisId;
                }

                if ($paidAt) {
                    $locked->qrispy_paid_at = $paidAt;
                } else {
                    $locked->qrispy_paid_at = Carbon::now();
                }

                $locked->save();

                $processedOrderId = $locked->order_id;

                return $locked;
            });

            if (!$updatedOrder) {
                Log::warning('[QRISPY] WEBHOOK_ORDER_NOT_FOUND', [
                    'order_id' => $orderId,
                    'payment_reference' => $paymentReference,
                    'unique_id' => (string) $uniqueId,
                ]);
                return response()->json(['success' => false, 'message' => 'Order not found'], 404);
            }

            Log::info('[QRISPY] WEBHOOK_ACCEPTED', [
                'order_id' => $updatedOrder->order_id,
                'payment_status' => $updatedOrder->payment_status,
                'buy_status' => $updatedOrder->buy_status,
                'unique_id' => (string) $uniqueId,
            ]);

            if ($processedOrderId) {
                DB::afterCommit(function () use ($processedOrderId) {
                    $o = Order::query()->where('order_id', $processedOrderId)->first();
                    if (!$o) return;

                    if ($o->type === 'Akun') {
                        $this->processAccountTransaction($o);
                    } else {
                        $this->processDigiflazzTransaction($o);
                    }

                    $o->refresh();

                    event(new PaymentStatusUpdated($o));
                    event(new RealtimeTransaction($o));
                });
            } else {
                event(new PaymentStatusUpdated($updatedOrder));
                event(new RealtimeTransaction($updatedOrder));
            }

            return response()->json(['success' => true]);
        } catch (\Throwable $e) {
            Log::error('[QRISPY] WEBHOOK_FAILED', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);

            return response()->json(['success' => false], 500);
        }
    }

    private function processDigiflazzTransaction(Order $order): void
    {
        app(DigiflazzService::class)->submit($order);
    }

    private function processAccountTransaction(Order $order): void
    {
        $product = Product::where('code', $order->code_product)->first();

        if (!$product) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Product not found',
            ]);
            return;
        }

        if ($product->type_transaction === 'manual') {
            $manualMessage = "Pesanan Anda sedang diproses secara manual.\nAdmin akan segera menghubungi Anda.\nHarap tunggu dan pastikan nomor WhatsApp Anda aktif.";

            $order->update([
                'serial_number' => $manualMessage,
                'buy_status' => 'Proses',
                'df_message' => 'Order will be processed manually',
            ]);
            return;
        }

        $account = Account::where('code_account', $order->code_product)
            ->where('stok', '>', 0)
            ->where('status', 'available')
            ->first();

        if (!$account) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Account not available',
                'serial_number' => 'Stok akun tidak tersedia, hubungi admin untuk mendapatkan bantuan!',
            ]);
            return;
        }

        $credentials = "Berikut detail akun Anda:\nEmail: {$account->email}\nPassword: {$account->password}\n";

        if ($account->descriptions) {
            $credentials .= "Keterangan: {$account->descriptions}\n";
        }

        $order->update([
            'serial_number' => $credentials,
            'buy_status' => 'Sukses',
            'df_message' => 'Account delivered successfully',
        ]);

        $account->decrement('stok');

        if ($account->stok <= 0) {
            $account->update(['status' => 'sold']);
        }
    }
}
