<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Services\PromoCodeService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class PromoController extends Controller
{
    public function index()
    {
        $products = Product::query()
            ->whereHas('activePromotion')
            ->with(['activePromotion', 'gameByBrand', 'logo'])
            ->get()
            ->map(function (Product $product) {
                $game = $product->gameByBrand;

                return [
                    'id' => $product->id,
                    'code' => $product->code,
                    'images' => $product->images ? asset('storage/' . $product->images) : null,
                    'banner' => $product->banner ? asset('storage/' . $product->banner) : null,
                    'title' => $product->title,
                    'brand' => $product->brand,
                    'logo' => $product->logo ? asset('storage/' . $product->logo->image) : null,
                    'type' => $product->type,
                    'category' => $product->category,
                    'selling_price' => $product->selling_price,
                    'selling_price_gold' => $product->selling_price_gold,
                    'selling_price_platinum' => $product->selling_price_platinum,
                    'promo_price' => $product->promo_price,
                    'is_promo' => $product->promo_price ? true : false,
                    'description' => $product->description,
                    'status' => $product->status,
                    'end_at' => optional($product->activePromotion)->end_at
                        ? Carbon::parse($product->activePromotion->end_at)->setTimezone('Asia/Jakarta')->toIso8601String()
                        : null,
                    'game_slug' => $game?->slug,
                    'game_image' => $game?->image ? asset("storage/{$game->image}") : null,
                ];
            })
            ->values();

        return response()->json([
            'success' => true,
            'products' => $products,
        ]);
    }

    public function validatePromo(Request $request, PromoCodeService $promoService)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:64',
            'game_slug' => 'required|string|max:255',
            'product_id' => 'required|integer',
            'payment_method_id' => 'required|integer',
            'whatsapp' => 'nullable|string|max:32',
        ]);

        $userId = auth()->check() ? auth()->id() : null;
        $role = auth()->check() ? (auth()->user()->role ?? null) : null;

        $code = strtoupper(trim($validated['code']));

        $result = $promoService->validatePromo(
            $code,
            $validated['game_slug'],
            (int) $validated['product_id'],
            (int) $validated['payment_method_id'],
            $role,
            $userId,
            $validated['whatsapp'] ?? null
        );

        if (!($result['valid'] ?? false)) {
            return response()->json([
                'success' => false,
                'message' => $result['message'] ?? 'Kode promo tidak valid.',
            ], 422);
        }

        return response()->json([
            'success' => true,
            'message' => $result['message'] ?? 'Kode promo dapat digunakan.',
            'data' => [
                'promo' => $result['promo'] ?? null,
                'pricing' => $result['pricing'] ?? null,
            ],
        ]);
    }
}