<?php

namespace App\Http\Controllers\Api;

use App\Events\RealtimeTransaction;
use App\Http\Controllers\Controller;
use App\Models\Game;
use App\Models\GameConfiguration;
use App\Models\Order;
use App\Models\PaymentMethod;
use App\Models\Product;
use App\Models\ProductPromotion;
use App\Models\PromoCode;
use App\Models\Setting;
use App\Models\QrisPayment;
use App\Services\PromoCodeService;
use App\Services\SmpQrisService;
use App\Services\QrispyService;
use App\Jobs\SendWhatsAppNotificationJob;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    protected ?string $tripayMerchant;
    protected ?string $tripayApiKey;
    protected ?string $tripayPrivateKey;

    protected ?string $duitkuMerchant;
    protected ?string $duitkuApiKey;

    protected string $digiflazzApiUrl;
    protected ?string $digiflazzUsername;
    protected ?string $digiflazzApiKey;

    protected ?string $fonnteToken;

    public function __construct()
    {
        $settings = Setting::all()->pluck('value', 'key');

        $this->tripayMerchant = $settings['tripay.merchant'] ?? null;
        $this->tripayApiKey = $settings['tripay.apikey'] ?? null;
        $this->tripayPrivateKey = $settings['tripay.privatekey'] ?? null;

        $this->duitkuMerchant = $settings['duitku.merchant'] ?? null;
        $this->duitkuApiKey = $settings['duitku.apikey'] ?? null;

        $this->digiflazzApiUrl = 'https://api.digiflazz.com/v1/transaction';
        $this->digiflazzUsername = $settings['digi.username'] ?? null;
        $this->digiflazzApiKey = $settings['digi.apikey'] ?? null;

        $this->fonnteToken = $settings['fonnte.token'] ?? null;
    }

    private function moneyToInt($value): int
    {
        if ($value === null) return 0;
        if (is_int($value)) return $value;
        if (is_float($value)) return (int) round($value);
        if (is_numeric($value) && !is_string($value)) return (int) $value;

        $s = trim((string) $value);
        if ($s === '') return 0;

        $digits = preg_replace('/[^0-9]/', '', $s);
        if ($digits === '' || $digits === null) return 0;

        return (int) $digits;
    }

    private function resolveActiveProductPromotionPrice(int $productId): ?int
    {
        $now = Carbon::now();

        $promo = ProductPromotion::where('product_id', $productId)
            ->where('is_active', true)
            ->where('start_at', '<=', $now)
            ->where('end_at', '>=', $now)
            ->orderByDesc('id')
            ->first();

        if (!$promo) return null;

        $price = $this->moneyToInt($promo->promo_price ?? 0);
        return $price > 0 ? $price : null;
    }

    public function index($slug)
    {
        try {
            $game = Game::where('slug', $slug)->firstOrFail();

            $products = Product::with(['logo', 'categoryRelation', 'activePromotion'])
                ->where('brand', $game->brand)
                ->where('status', true)
                ->orderBy('selling_price', 'asc')
                ->get()
                ->map(function ($product) {
                    $promo = $product->activePromotion;

                    return [
                        'id' => $product->id,
                        'code' => $product->code,
                        'images' => $product->images ? asset('storage/' . $product->images) : null,
                        'banner' => $product->banner ? asset('storage/' . $product->banner) : null,
                        'title' => $product->title,
                        'brand' => $product->brand,
                        'logo' => $product->logo ? asset('storage/' . $product->logo->image) : null,
                        'type' => $product->type,
                        'category' => $product->categoryRelation ? [
                            'id' => $product->categoryRelation->id,
                            'title' => $product->categoryRelation->title,
                            'logo' => $product->categoryRelation->logo ? asset('storage/' . $product->categoryRelation->logo) : null,
                            'game' => $product->categoryRelation->game,
                        ] : null,
                        'selling_price' => $product->selling_price,
                        'selling_price_gold' => $product->selling_price_gold,
                        'selling_price_platinum' => $product->selling_price_platinum,
                        'promo_price' => $promo?->promo_price,
                        'is_promo' => $promo !== null,
                        'description' => $product->description,
                        'status' => $product->status,
                    ];
                });

            $gameConfiguration = GameConfiguration::where('game_name', $game->title)->first();

            $paymentMethod = PaymentMethod::all();
            foreach ($paymentMethod as $payment) {
                $payment->images = $payment->images ? asset("storage/{$payment->images}") : null;
            }

            return response()->json([
                'success' => true,
                'game' => [
                    'id' => $game->id,
                    'title' => $game->title,
                    'slug' => $game->slug,
                    'brand' => $game->brand,
                    'developers' => $game->developers,
                    'image' => $game->image ? asset('storage/' . $game->image) : null,
                    'banner' => $game->banner ? asset('storage/' . $game->banner) : null,
                    'description' => $game->description,
                    'populer' => $game->populer,
                    'status' => $game->status,
                ],
                'products' => $products,
                'gameConfiguration' => $gameConfiguration,
                'paymentMethod' => $paymentMethod,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Game not found',
                'error' => $e->getMessage(),
            ], 404);
        }
    }

    public function store(Request $request)
    {
        $trace = 'INV-' . now()->format('YmdHis') . '-' . substr(md5((string) microtime(true)), 0, 8);

        Log::info('[ORDER] HIT', [
            'trace' => $trace,
            'path' => $request->path(),
            'ip' => $request->ip(),
            'payload' => $request->all(),
        ]);

        $validator = Validator::make($request->all(), [
            'id' => 'required|string',
            'server' => 'nullable|string',
            'game' => 'required|string',
            'product_id' => 'required|integer|exists:products,id',
            'payment_method_id' => 'required|integer|exists:payment_methods,id',
            'email' => 'nullable|email',
            'whatsapp' => 'required|string',
            'nickname' => 'nullable|string',
            'promo_code' => 'nullable|string|max:64|regex:/^[A-Za-z0-9_-]+$/',
        ]);

        if ($validator->fails()) {
            Log::warning('[ORDER] VALIDATION_FAILED', [
                'trace' => $trace,
                'errors' => $validator->errors()->toArray(),
            ]);

            return response()->json([
                'message' => 'Validasi gagal.',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $userId = auth()->check() ? auth()->id() : null;
            $user = auth()->user();
            $role = $user?->role;

            $id_game = $request->input('id');
            $server_game = $request->input('server');
            $game_slug = $request->input('game');
            $product_id = (int) $request->input('product_id');
            $payment_method_id = (int) $request->input('payment_method_id');
            $email = $request->input('email');
            $whatsapp = preg_replace('/[^0-9+]/', '', (string) $request->input('whatsapp'));
            $nickname = $request->input('nickname');

            $promoCodeInput = $request->input('promo_code');
            $promoCodeInput = $promoCodeInput ? strtoupper(trim((string) $promoCodeInput)) : null;

            Log::info('[ORDER] INPUT_PARSED', [
                'trace' => $trace,
                'user_id' => $userId,
                'role' => $role,
                'game_slug' => $game_slug,
                'product_id' => $product_id,
                'payment_method_id' => $payment_method_id,
            ]);

            $game = Game::where('slug', $game_slug)->firstOrFail();
            $product = Product::findOrFail($product_id);
            $paymentMethod = PaymentMethod::findOrFail($payment_method_id);

            Log::info('[ORDER] MODELS_LOADED', [
                'trace' => $trace,
                'game' => $game->title,
                'product' => $product->title,
                'product_code' => $product->code,
                'payment_provider' => $paymentMethod->provider,
                'payment_code' => $paymentMethod->code,
            ]);

            $orderId = 'INV-' . time();

            switch ($role) {
                case 'platinum':
                    $selling_price = $this->moneyToInt($product->selling_price_platinum ?? $product->selling_price);
                    $profit = $this->moneyToInt($product->profit_platinum ?? $product->profit);
                    break;
                case 'gold':
                    $selling_price = $this->moneyToInt($product->selling_price_gold ?? $product->selling_price);
                    $profit = $this->moneyToInt($product->profit_gold ?? $product->profit);
                    break;
                default:
                    $selling_price = $this->moneyToInt($product->selling_price);
                    $profit = $this->moneyToInt($product->profit);
                    break;
            }

            $promoPrice = $this->resolveActiveProductPromotionPrice((int) $product->id);
            $final_price = (int) ($promoPrice ?? $selling_price);
            $final_price = max(0, $final_price);

            $promoId = null;
            $promoDiscount = 0;

            if ($promoCodeInput) {
                $promoService = app(PromoCodeService::class);
                $promoResult = $promoService->validatePromo(
                    $promoCodeInput,
                    $game_slug,
                    $product_id,
                    $payment_method_id,
                    $role,
                    $userId,
                    $whatsapp
                );

                if (!($promoResult['valid'] ?? false)) {
                    return response()->json([
                        'success' => false,
                        'message' => $promoResult['message'] ?? 'Kode promo tidak valid.',
                    ], 422);
                }

                $promoId = (int) ($promoResult['promo']['id'] ?? 0);
                $promoDiscount = (int) ($promoResult['pricing']['discount'] ?? 0);
                $serviceFinal = (int) ($promoResult['pricing']['final_price'] ?? 0);

                if ($serviceFinal > 0) {
                    $final_price = $serviceFinal;
                } elseif ($promoDiscount > 0) {
                    $final_price = max(0, (int) $final_price - (int) $promoDiscount);
                }
            }

            Log::info('[ORDER] PRICE_RESOLVED', [
                'trace' => $trace,
                'order_id' => $orderId,
                'selling_price' => $selling_price,
                'promo_price' => $promoPrice,
                'final_price' => $final_price,
                'profit' => $profit,
                'promo_discount' => $promoDiscount,
            ]);

            $responseData = null;

            if ($paymentMethod->provider === 'tripay') {
                Log::info('[ORDER] CREATE_PAYMENT_TRIPAY', ['trace' => $trace, 'order_id' => $orderId]);
                $responseData = $this->createTripayTransaction($paymentMethod, $orderId, $product, $final_price, $email, $whatsapp);
            } elseif ($paymentMethod->provider === 'paydisini') {
                Log::info('[ORDER] CREATE_PAYMENT_PAYDISINI', ['trace' => $trace, 'order_id' => $orderId]);
                $responseData = $this->createPaydisiniTransaction($orderId, $final_price, $paymentMethod->code, $email, $whatsapp, $product->title);
            } elseif ($paymentMethod->provider === 'duitku') {
                Log::info('[ORDER] CREATE_PAYMENT_DUITKU', ['trace' => $trace, 'order_id' => $orderId]);
                $responseData = $this->createDuitkuTransaction($paymentMethod, $orderId, $product, $final_price, $email, $whatsapp);
            } elseif ($paymentMethod->provider === 'qrispy') {
                Log::info('[ORDER] CREATE_PAYMENT_QRISPY', ['trace' => $trace, 'order_id' => $orderId]);

                $settings = Setting::query()->pluck('value', 'key')->all();
                $enabled = filter_var($settings['qrispy.enabled'] ?? false, FILTER_VALIDATE_BOOLEAN);

                if (!$enabled) {
                    return response()->json(['message' => 'Metode pembayaran QRISPY sedang tidak tersedia.'], 400);
                }

                $referencePrefix = trim((string) ($settings['qrispy.reference_prefix'] ?? 'Order-'));
                if ($referencePrefix === '') {
                    $referencePrefix = 'Order-';
                }

                $paymentReference = $referencePrefix . $orderId;

                $feeFixed = $this->moneyToInt($paymentMethod->fee ?? 0);
                $feePercent = (float) ($paymentMethod->fee_percent ?? 0);
                $feePercentAmount = (int) ceil($final_price * ($feePercent / 100));
                $fee = max(0, $feeFixed + $feePercentAmount);

                $amountTotal = (int) max(0, (int) $final_price + $fee);

                if ($amountTotal < 1000) {
                    return response()->json([
                        'message' => 'Minimal pembayaran QRISPY adalah Rp 1.000.',
                    ], 422);
                }

                $qrispy = app(QrispyService::class);
                $qrispyData = $qrispy->generate($amountTotal, $paymentReference);

                $qrisId = trim((string) ($qrispyData['qris_id'] ?? ''));
                $qrisImageUrl = trim((string) ($qrispyData['qris_image_url'] ?? ''));
                $qrisImageBase64 = trim((string) ($qrispyData['qris_image_base64'] ?? ''));

                if ($qrisId === '') {
                    return response()->json(['message' => 'QRISPY gagal membuat QR (qris_id kosong).'], 500);
                }

                if ($qrisImageUrl === '' && $qrisImageBase64 === '') {
                    return response()->json(['message' => 'QRISPY gagal membuat QR (gambar QR kosong).'], 500);
                }

                $expiresAt = null;

                if (!empty($qrispyData['expired_at'])) {
                    try {
                        $expiresAt = Carbon::parse((string) $qrispyData['expired_at']);
                    } catch (\Throwable $e) {
                        $expiresAt = null;
                    }
                }

                if (!$expiresAt) {
                    $expiresIn = (int) ($qrispyData['expires_in_seconds'] ?? 0);
                    if ($expiresIn > 0) {
                        $expiresAt = Carbon::now()->addSeconds($expiresIn);
                    } else {
                        $cfgExpiry = (int) ($settings['qrispy.expiry_seconds'] ?? 900);
                        $cfgExpiry = max(60, min(86400, $cfgExpiry));
                        $expiresAt = Carbon::now()->addSeconds($cfgExpiry);
                    }
                }

                $useBase64 = filter_var($settings['qrispy.use_base64'] ?? false, FILTER_VALIDATE_BOOLEAN);
                $displayName = (string) ($settings['qrispy.display_name'] ?? ($paymentMethod->name ?? 'QRIS'));

                $responseData = [
                    'success' => true,
                    'data' => [
                        'amount' => $amountTotal,
                        'fee' => $fee,
                        'payment_name' => $displayName,
                        'payment_code' => $qrisImageUrl !== '' ? $qrisImageUrl : 'Scan QR di halaman invoice',
                        'expired_time' => $expiresAt->timestamp,
                        'expires_at' => $expiresAt->toDateTimeString(),
                        'qris_id' => $qrisId,
                        'qris_image_url' => $qrisImageUrl !== '' ? $qrisImageUrl : null,
                        'qris_image_base64' => $useBase64 && $qrisImageBase64 !== '' ? $qrisImageBase64 : null,
                        'payment_reference' => (string) ($qrispyData['payment_reference'] ?? $paymentReference),
                    ],
                ];
            } elseif ($paymentMethod->provider === 'smp') {
                Log::info('[ORDER] CREATE_PAYMENT_SMP', ['trace' => $trace, 'order_id' => $orderId]);

                $smp = app(SmpQrisService::class);
                $smpData = $smp->create($paymentMethod, (int) $final_price);

                $responseData = [
                    'success' => true,
                    'data' => [
                        'amount' => $smpData['amount'],
                        'fee' => $smpData['fee'],
                        'payment_name' => $smpData['payment_name'],
                        'qrString' => $smpData['qrString'],
                        'expired_time' => $smpData['expired_time'],
                        'expires_at' => $smpData['expires_at']->toDateTimeString(),
                        'unique_code' => $smpData['unique_code'],
                    ],
                ];
            } else {
                Log::warning('[ORDER] UNKNOWN_PROVIDER', ['trace' => $trace, 'provider' => $paymentMethod->provider]);
                return response()->json(['message' => 'Metode pembayaran tidak dikenali.'], 400);
            }

            Log::info('[ORDER] PAYMENT_RESPONSE', ['trace' => $trace, 'response' => $responseData]);

            if (!($responseData['success'] ?? false)) {
                Log::warning('[ORDER] PAYMENT_FAILED', [
                    'trace' => $trace,
                    'message' => $responseData['message'] ?? null,
                    'response' => $responseData
                ]);
                return response()->json(['message' => $responseData['message'] ?? 'Gagal membuat transaksi.'], 500);
            }

            $data = $responseData['data'] ?? [];
            $amount = $this->moneyToInt($data['amount'] ?? 0);

            if ($amount <= 0) {
                Log::error('[ORDER] AMOUNT_MISSING', ['trace' => $trace, 'data' => $data]);
                return response()->json(['message' => 'Nominal pembayaran tidak ditemukan.'], 500);
            }

            $paymentCode = $data['payment_code']
                ?? $data['qr_url']
                ?? $data['pay_code']
                ?? $data['pay_url']
                ?? $data['qrcode_url']
                ?? $data['virtual_account']
                ?? $data['checkout_url']
                ?? $data['vaNumber']
                ?? $data['qrString']
                ?? $data['paymentUrl']
                ?? null;

            if (!$paymentCode) {
                Log::error('[ORDER] PAYMENT_CODE_MISSING', ['trace' => $trace, 'data' => $data]);
                return response()->json(['message' => 'Kode pembayaran tidak ditemukan.'], 500);
            }

            $productPrice = $this->moneyToInt($final_price);
            $duitkuFee = max(0, $amount - $productPrice);
            $fee = $this->moneyToInt($data['fee'] ?? $data['total_fee'] ?? $duitkuFee);

            $expiredTime = is_numeric($data['expired_time'] ?? null)
                ? (int) $data['expired_time']
                : (!empty($data['expired']) ? (int) strtotime($data['expired']) : (int) strtotime('+60 minutes'));

            Log::info('[ORDER] CREATING_ORDER', ['trace' => $trace, 'order_id' => $orderId]);

            $order = DB::transaction(function () use (
                $promoId,
                $promoCodeInput,
                $promoDiscount,
                $userId,
                $orderId,
                $game,
                $product,
                $id_game,
                $server_game,
                $nickname,
                $email,
                $whatsapp,
                $final_price,
                $profit,
                $fee,
                $amount,
                $data,
                $paymentMethod,
                $paymentCode,
                $expiredTime,
                $game_slug,
                $product_id,
                $payment_method_id,
                $role,
                $promoPrice
            ) {
                $appliedPromoId = null;
                $appliedPromoCode = null;
                $appliedPromoDiscount = null;

                if ($promoId && $promoCodeInput) {
                    $promo = PromoCode::query()->lockForUpdate()->whereKey($promoId)->first();

                    if ($promo) {
                        $promoService = app(PromoCodeService::class);
                        $recheck = $promoService->validatePromo(
                            $promoCodeInput,
                            $game_slug,
                            $product_id,
                            $payment_method_id,
                            $role,
                            $userId,
                            $whatsapp
                        );

                        if (($recheck['valid'] ?? false) && (int) ($recheck['promo']['id'] ?? 0) === $promoId) {
                            $totalOk = true;
                            if ($promo->usage_limit_total !== null) {
                                $totalOk = (int) $promo->used_count < (int) $promo->usage_limit_total;
                            }

                            if ($totalOk) {
                                $promo->increment('used_count');
                                $appliedPromoId = $promoId;
                                $appliedPromoCode = $promoCodeInput;
                                $appliedPromoDiscount = $promoDiscount;
                            }
                        }
                    }
                }

                $order = Order::create([
                    'user_id' => $userId,
                    'order_id' => $orderId,
                    'games' => $game->title,
                    'product' => $product->title,
                    'type' => $product->type,
                    'code_product' => $product->code,
                    'id_games' => $id_game,
                    'server_games' => $server_game,
                    'nickname' => $nickname,
                    'email' => $email,
                    'whatsapp' => $whatsapp,
                    'price' => $final_price,
                    'discount_price' => $promoPrice,
                    'promo_code_id' => $appliedPromoId,
                    'promo_code' => $appliedPromoCode,
                    'promo_discount' => $appliedPromoDiscount,
                    'profit' => $profit,
                    'fee' => $fee,
                    'total_price' => $amount,
                    'payment_name' => $data['payment_name'] ?? $data['service_name'] ?? $paymentMethod->name,
                    'payment_method' => $paymentMethod->code,
                    'payment_code' => $paymentCode,
                    'qrispy_qris_id' => $data['qris_id'] ?? null,
                    'qrispy_image_url' => $data['qris_image_url'] ?? null,
                    'qrispy_image_base64' => $data['qris_image_base64'] ?? null,
                    'qrispy_expires_at' => $paymentMethod->provider === 'qrispy' && !empty($data['expires_at']) ? Carbon::parse($data['expires_at']) : null,
                    'expired_time' => $expiredTime,
                    'payment_status' => 'UNPAID',
                    'payment_instructions' => $data['instructions'] ?? $data['payment_guide'] ?? null,
                    'buy_status' => 'Pending',
                ]);

                if ($paymentMethod->provider === 'smp') {
                    $expiresAt = !empty($data['expires_at']) ? Carbon::parse($data['expires_at']) : Carbon::now()->addMinutes(60);

                    QrisPayment::create([
                        'order_pk' => $order->id,
                        'order_id' => $orderId,
                        'amount_base' => (int) $final_price,
                        'unique_code' => (int) ($data['unique_code'] ?? 0),
                        'amount_total' => $amount,
                        'status' => 'pending',
                        'expires_at' => $expiresAt,
                        'meta' => [
                            'provider' => 'smp',
                            'payment_method_id' => $paymentMethod->id,
                            'qrString' => $data['qrString'] ?? null,
                        ],
                    ]);
                }

                return $order;
            });

            Log::info('[ORDER] ORDER_CREATED', ['trace' => $trace, 'order_pk' => $order->id, 'order_id' => $order->order_id]);

            event(new RealtimeTransaction($order));

            $response = response()->json([
                'success' => true,
                'message' => 'Pesanan Berhasil Dibuat!',
                'orderId' => $orderId,
                'data' => [
                    'order_id' => $orderId,
                    'payment_method' => $paymentMethod->code,
                    'payment_name' => $data['payment_name'] ?? $data['service_name'] ?? $paymentMethod->name,
                    'payment_code' => $paymentCode,
                    'total_price' => $amount,
                    'expired_time' => (int) ($order->expired_time ?? 0),
                    'payment_status' => 'UNPAID',
                    'buy_status' => 'Pending',
                    'invoice_url' => config('app.front_url') . "/invoices/{$orderId}",
                ],
            ], 201);

            $message = "*Menunggu Pembayaran*\n";
            $message .= "No. Invoice: *{$orderId}*\n\n";
            $message .= "=======================\n";
            $message .= "*Detail Pembelian*\n\n";
            $message .= "*Produk:* {$game->title}\n";
            $message .= "*Item:* {$product->title}\n";
            $message .= "*Harga:* Rp " . number_format($final_price, 0, ',', '.') . "\n";
            $message .= "*Fee:* Rp " . number_format($fee, 0, ',', '.') . "\n";
            $message .= "*Total Bayar:* Rp " . number_format($amount, 0, ',', '.') . "\n";
            $message .= "*Status:* Menunggu Pembayaran\n\n";
            $message .= "=======================\n";
            if ($paymentMethod->provider === 'smp' || $paymentMethod->provider === 'qrispy') {
                $message .= "*Cara bayar:* Scan QR di halaman invoice\n\n";
            }
            $message .= "*Invoice:*\n" . config('app.front_url') . "/invoices/{$orderId}\n\n";
            $message .= "_Terima kasih_";

            Log::info('[ORDER] SENDING_WA_QUEUED', [
                'trace' => $trace,
                'order_id' => $orderId,
                'whatsapp' => $whatsapp,
            ]);

            SendWhatsAppNotificationJob::dispatch(
                $whatsapp,
                'waiting_payment',
                [
                    'order_id' => $orderId,
                    'games' => $game->title,
                    'product' => $product->title,
                    'nickname' => $nickname,
                    'id_games' => $id_game,
                    'server_games' => $server_game,
                    'price' => $final_price,
                    'fee' => $fee,
                    'total_price' => $amount,
                    'payment_name' => $paymentMethod->name,
                    'payment_method' => $paymentMethod->code,
                    'payment_code' => in_array($paymentMethod->provider, ['smp', 'qrispy'], true) ? 'Scan QR di halaman invoice' : $paymentCode,
                    'payment_status' => 'UNPAID',
                    'buy_status' => 'Pending',
                    'expired_time' => (int) ($order->expired_time ?? 0),
                    'invoice_url' => config('app.front_url') . "/invoices/{$orderId}",
                    'app_name' => config('app.name'),
                ],
                $message
            )->afterResponse();

            Log::info('[ORDER] DONE', [
                'trace' => $trace,
                'order_id' => $orderId,
            ]);

            return $response;
        } catch (\Throwable $e) {
            Log::error('[ORDER] FAILED', [
                'trace' => $trace,
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);

            return response()->json([
                'message' => 'Terjadi kesalahan saat memproses pesanan.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    protected function createTripayTransaction($paymentMethod, $orderId, $product, $final_price, $email, $whatsapp)
    {
        $merchantCode = $this->tripayMerchant;
        $privateKey = $this->tripayPrivateKey;
        $apiKey = $this->tripayApiKey;

        $payload = [
            'method' => $paymentMethod->code,
            'merchant_ref' => $orderId,
            'amount' => $final_price,
            'customer_name' => env('APP_NAME'),
            'customer_email' => 'default@email.io',
            'customer_phone' => $whatsapp,
            'order_items' => [
                [
                    'sku' => 'PROD-' . $product->id,
                    'name' => $product->title,
                    'price' => $final_price,
                    'quantity' => 1,
                ],
            ],
            'return_url' => '',
            'expired_time' => time() + (24 * 60 * 60),
            'signature' => hash_hmac('sha256', $merchantCode . $orderId . $final_price, $privateKey),
        ];

        $client = new \GuzzleHttp\Client();
        $response = $client->post('https://tripay.co.id/api/transaction/create', [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Accept' => 'application/json',
            ],
            'json' => $payload,
        ]);

        $responseData = json_decode($response->getBody(), true);

        return [
            'success' => $responseData['success'] ?? false,
            'data' => $responseData['data'] ?? null,
            'message' => $responseData['message'] ?? null,
        ];
    }

    protected function createPaydisiniTransaction($orderId, $final_price, $paymentMethodCode, $email, $whatsapp, $productTitle)
    {
        $apiKey = config('services.paydisini.api_key');
        $validTime = 1800;

        $payload = [
            'key' => $apiKey,
            'request' => 'new',
            'unique_code' => $orderId,
            'service' => $paymentMethodCode,
            'amount' => $final_price,
            'note' => 'Topup ' . $productTitle,
            'valid_time' => $validTime,
            'type_fee' => 1,
            'payment_guide' => true,
            'customer_email' => $email,
            'ewallet_phone' => $whatsapp,
            'signature' => md5($apiKey . $orderId . $paymentMethodCode . $final_price . $validTime . 'NewTransaction'),
        ];

        try {
            $response = Http::asForm()->post('https://api.paydisini.co.id/v1/', $payload);
            $json = $response->json();

            if ($response->successful() && ($json['success'] ?? false) === true) {
                $data = $json['data'] ?? [];

                if (empty($data)) {
                    Log::error('Response Paydisini tidak berisi data transaksi.', [
                        'order_id' => $orderId,
                        'response' => $json,
                    ]);

                    return [
                        'success' => false,
                        'message' => 'Response dari PayDisini tidak berisi data transaksi.',
                    ];
                }

                return [
                    'success' => true,
                    'data' => $data,
                ];
            }

            return [
                'success' => false,
                'message' => $json['message'] ?? 'Gagal membuat transaksi di PayDisini',
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan saat menghubungi PayDisini: ' . $e->getMessage(),
            ];
        }
    }

    protected function createDuitkuTransaction($paymentMethod, $orderId, $product, $final_price, $email, $whatsapp)
    {
        $merchantCode = $this->duitkuMerchant;
        $apiKey = $this->duitkuApiKey;
        $callbackUrl = env('APP_URL') . '/api/duitku/callback';
        $returnUrl = env('FRONT_URL') . "/invoices/{$orderId}";

        $codesWithFee = ['SP', 'NQ', 'DQ', 'GQ', 'SQ'];
        if (in_array($paymentMethod->code, $codesWithFee)) {
            $final_price += $final_price * 0.007;
            $final_price = round($final_price);
        }

        $signature = md5($merchantCode . $orderId . $final_price . $apiKey);

        $payload = [
            'merchantCode' => $merchantCode,
            'paymentAmount' => $final_price,
            'paymentMethod' => $paymentMethod->code,
            'merchantOrderId' => $orderId,
            'productDetails' => $product->title,
            'email' => 'default@email.io',
            'phoneNumber' => $whatsapp,
            'itemDetails' => [
                [
                    'name' => $product->title,
                    'price' => $final_price,
                    'quantity' => 1,
                ],
            ],
            'customerVaName' => env('APP_NAME'),
            'callbackUrl' => $callbackUrl,
            'returnUrl' => $returnUrl,
            'expiryPeriod' => 60,
            'signature' => $signature,
        ];

        $client = new \GuzzleHttp\Client();
        $response = $client->post('https://passport.duitku.com/webapi/api/merchant/v2/inquiry', [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'json' => $payload,
        ]);

        $responseData = json_decode($response->getBody(), true);

        return [
            'success' => ($responseData['statusCode'] ?? null) == '00',
            'data' => $responseData,
            'message' => $responseData['statusMessage'] ?? 'Failed to create transaction',
        ];
    }

    private function sendWhatsAppNotification(string $whatsapp, string $message, string $templateKey, array $vars = []): void
    {
        $vars['app_name'] = config('app.name');
        $vars['invoice_url'] = config('app.front_url') . "/invoices/{$vars['order_id']}";

        app(\App\Services\WhatsAppNotificationService::class)->send($whatsapp, $templateKey, $vars, $message);
    }

    public function show($orderId)
    {
        $order = Order::where('order_id', $orderId)->first();

        if (!$order) {
            return response()->json(['message' => 'Pesanan tidak ditemukan'], 404);
        }

        $game = Game::where('title', $order->games)->firstOrFail();

        $gameData = $game->toArray();
        $gameData['image'] = $game->image ? asset("storage/{$game->image}") : null;
        $gameData['banner'] = $game->banner ? asset("storage/{$game->banner}") : null;

        $qris = null;

        if ($order->payment_method && str_starts_with(strtoupper($order->payment_method), 'SMP')) {
            $qrisPayment = QrisPayment::where('order_id', $order->order_id)->first();

            if ($qrisPayment) {
                $qrisString = data_get($qrisPayment->meta, 'qrString');

                if (!$qrisString) {
                    $settings = Setting::query()->pluck('value', 'key')->all();
                    $qrisString = $settings['smp.qris_string'] ?? null;
                }

                $qris = [
                    'provider' => 'smp',
                    'qrString' => $qrisString,
                    'expires_at' => optional($qrisPayment->expires_at)->toDateTimeString(),
                    'status' => $qrisPayment->status,
                    'amount_total' => $qrisPayment->amount_total,
                ];
            }
        }

        return response()->json([
            'success' => true,
            'game' => $gameData,
            'order' => $order->only([
                'order_id',
                'games',
                'product',
                'id_games',
                'server_games',
                'nickname',
                'price',
                'discount_price',
                'promo_code',
                'promo_discount',
                'fee',
                'total_price',
                'payment_name',
                'payment_method',
                'payment_code',
                'expired_time',
                'payment_status',
                'payment_instructions',
                'buy_status',
                'serial_number',
                'qrispy_qris_id',
                'qrispy_image_url',
                'qrispy_image_base64',
                'qrispy_expires_at',
                'qrispy_paid_at',
                'created_at',
            ]),
            'qris' => $qris,
        ]);
    }
}