<?php

namespace App\Http\Controllers\Api;

use App\Events\PaymentStatusUpdated;
use App\Events\RealtimeTransaction;
use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\Game;
use App\Models\Order;
use App\Models\PaymentMethod;
use App\Models\Product;
use App\Services\DigiflazzService;
use App\Services\QrispyService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class InvoiceController extends Controller
{
    private function normalizeWhatsapp(?string $value): ?string
    {
        if (!$value) return null;
        $v = preg_replace('/[^0-9]/', '', $value);
        if ($v === '') return null;
        if (str_starts_with($v, '0')) $v = '62' . substr($v, 1);
        if (!str_starts_with($v, '62')) $v = '62' . $v;
        return $v;
    }

    private function mapGame(?Order $order): ?array
    {
        if (!$order || !$order->games) return null;

        $g = Game::query()->where('title', $order->games)->first();
        if (!$g) return null;

        return [
            'id' => $g->id,
            'title' => $g->title,
            'slug' => $g->slug,
            'brand' => $g->brand,
            'developers' => $g->developers,
            'image' => $g->image ? asset('storage/' . $g->image) : null,
            'banner' => $g->banner ? asset('storage/' . $g->banner) : null,
            'description' => $g->description,
            'populer' => $g->populer,
            'status' => $g->status,
        ];
    }

    public function show(string $order_id)
    {
        $transaction = Order::query()->where('order_id', $order_id)->first();

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Invoice tidak ditemukan.',
            ], 404);
        }

        // Sinkronkan nama metode pembayaran dari tabel payment_methods (biar tidak ikut nama merchant)
        $code = strtoupper((string) ($transaction->payment_method ?? ''));
        if ($code !== '') {
            $pm = PaymentMethod::query()->where('code', $code)->first();
            if ($pm && $pm->name) {
                $transaction->setAttribute('payment_name', $pm->name);
            }
        }

        return response()->json([
            'success' => true,
            'order' => $transaction,
            'game' => $this->mapGame($transaction),
        ]);
    }

    public function paymentStatus(string $order_id)
    {
        $order = Order::query()->where('order_id', $order_id)->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Invoice tidak ditemukan.',
            ], 404);
        }

        if ($order->payment_status !== 'UNPAID') {
            return response()->json([
                'success' => true,
                'data' => $this->statusPayload($order),
            ]);
        }

        if (!$order->qrispy_qris_id) {
            return response()->json([
                'success' => true,
                'data' => $this->statusPayload($order),
            ]);
        }

        $expiredTime = (int) ($order->expired_time ?? 0);

        if ($expiredTime > 0 && time() >= $expiredTime) {
            $updated = $this->markOrderExpired($order_id);

            if ($updated) {
                event(new PaymentStatusUpdated($updated));
                event(new RealtimeTransaction($updated));
            }

            return response()->json([
                'success' => true,
                'data' => $this->statusPayload($updated ?? $order),
            ]);
        }

        $status = null;

        try {
            $status = app(QrispyService::class)->status((string) $order->qrispy_qris_id);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => true,
                'data' => $this->statusPayload($order),
            ]);
        }

        $normalized = $this->normalizeQrispyStatus($status);

        if (($normalized['state'] ?? 'UNPAID') === 'PAID') {
            $receivedAmount = (int) ($normalized['received_amount'] ?? 0);
            $totalPrice = (int) ($order->total_price ?? 0);

            if ($receivedAmount > 0 && $totalPrice > 0 && $receivedAmount < $totalPrice) {
                return response()->json([
                    'success' => true,
                    'data' => $this->statusPayload($order),
                ]);
            }

            $processedOrderId = null;

            $updatedOrder = DB::transaction(function () use ($order_id, $normalized, &$processedOrderId) {
                $locked = Order::query()->lockForUpdate()->where('order_id', $order_id)->first();

                if (!$locked) {
                    return null;
                }

                if ($locked->payment_status !== 'UNPAID') {
                    return $locked;
                }

                $paidAt = null;
                if (!empty($normalized['paid_at'])) {
                    try {
                        $paidAt = Carbon::parse((string) $normalized['paid_at']);
                    } catch (\Throwable $e) {
                        $paidAt = null;
                    }
                }

                $locked->payment_status = 'PAID';
                $locked->buy_status = 'Proses';
                $locked->qrispy_paid_at = $paidAt ?? Carbon::now();
                $locked->save();

                $processedOrderId = $locked->order_id;

                return $locked;
            });

            if ($processedOrderId) {
                DB::afterCommit(function () use ($processedOrderId) {
                    $o = Order::where('order_id', $processedOrderId)->first();

                    if (!$o) {
                        return;
                    }

                    if ($o->type === 'Akun') {
                        $this->processAccountTransaction($o);
                    } else {
                        $this->processDigiflazzTransaction($o);
                    }

                    $o->refresh();

                    event(new PaymentStatusUpdated($o));
                    event(new RealtimeTransaction($o));
                });
            } elseif ($updatedOrder) {
                event(new PaymentStatusUpdated($updatedOrder));
                event(new RealtimeTransaction($updatedOrder));
            }

            return response()->json([
                'success' => true,
                'data' => $this->statusPayload($updatedOrder ?? $order),
            ]);
        }

        if (($normalized['state'] ?? 'UNPAID') === 'EXPIRED') {
            $updated = $this->markOrderExpired($order_id);

            if ($updated) {
                event(new PaymentStatusUpdated($updated));
                event(new RealtimeTransaction($updated));
            }

            return response()->json([
                'success' => true,
                'data' => $this->statusPayload($updated ?? $order),
            ]);
        }

        return response()->json([
            'success' => true,
            'data' => $this->statusPayload($order),
        ]);
    }

    private function normalizeQrispyStatus(array $payload): array
    {
        $data = $payload['data'] ?? [];

        $status = strtolower((string) (
            $data['status']
                ?? $data['payment_status']
                ?? $data['state']
                ?? ''
        ));

        $paidAt = $data['paid_at'] ?? $data['paidAt'] ?? null;
        $receivedAmount = $data['received_amount'] ?? $data['receivedAmount'] ?? $data['amount_received'] ?? null;

        $isPaid = (bool) ($data['paid'] ?? false);
        $isExpired = (bool) ($data['expired'] ?? false);

        if ($isPaid || (!empty($paidAt)) || in_array($status, ['paid', 'success', 'settled', 'completed', 'received'], true)) {
            return [
                'state' => 'PAID',
                'paid_at' => $paidAt,
                'received_amount' => is_numeric($receivedAmount) ? (int) $receivedAmount : null,
            ];
        }

        if ($isExpired || in_array($status, ['expired', 'failed', 'cancelled', 'canceled', 'timeout'], true)) {
            return [
                'state' => 'EXPIRED',
                'paid_at' => $paidAt,
                'received_amount' => is_numeric($receivedAmount) ? (int) $receivedAmount : null,
            ];
        }

        return [
            'state' => 'UNPAID',
            'paid_at' => $paidAt,
            'received_amount' => is_numeric($receivedAmount) ? (int) $receivedAmount : null,
        ];
    }

    private function markOrderExpired(string $orderId): ?Order
    {
        $updated = null;

        DB::transaction(function () use ($orderId, &$updated) {
            $o = Order::query()->lockForUpdate()->where('order_id', $orderId)->first();

            if (!$o) {
                return;
            }

            if ($o->payment_status !== 'UNPAID') {
                $updated = $o;
                return;
            }

            $o->payment_status = 'EXPIRED';
            $o->buy_status = 'Batal';
            $o->save();

            $updated = $o;
        });

        return $updated;
    }

    private function statusPayload(Order $order): array
    {
        return [
            'order_id' => $order->order_id,
            'payment_status' => $order->payment_status,
            'buy_status' => $order->buy_status,
            'expired_time' => (int) ($order->expired_time ?? 0),
            'qrispy_qris_id' => $order->qrispy_qris_id,
            'qrispy_expires_at' => $order->qrispy_expires_at?->toDateTimeString(),
            'qrispy_paid_at' => $order->qrispy_paid_at?->toDateTimeString(),
        ];
    }

    private function processDigiflazzTransaction(Order $order): void
    {
        app(DigiflazzService::class)->submit($order);
    }

    private function processAccountTransaction(Order $order): void
    {
        $product = Product::where('code', $order->code_product)->first();

        if (!$product) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Product not found',
            ]);

            return;
        }

        if ($product->type_transaction === 'manual') {
            $manualMessage = "Pesanan Anda sedang diproses secara manual.\nAdmin akan segera menghubungi Anda.\nHarap tunggu dan pastikan nomor WhatsApp Anda aktif.";

            $order->update([
                'serial_number' => $manualMessage,
                'buy_status' => 'Proses',
                'df_message' => 'Order will be processed manually',
            ]);

            return;
        }

        $account = Account::where('code_account', $order->code_product)
            ->where('stok', '>', 0)
            ->where('status', 'available')
            ->first();

        if (!$account) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Account not available',
                'serial_number' => 'Stok akun tidak tersedia, hubungi admin untuk mendapatkan bantuan!',
            ]);

            return;
        }

        $credentials = "Berikut detail akun Anda:\nEmail: {$account->email}\nPassword: {$account->password}\n";

        if ($account->descriptions) {
            $credentials .= "Keterangan: {$account->descriptions}\n";
        }

        $order->update([
            'serial_number' => $credentials,
            'buy_status' => 'Sukses',
            'df_message' => 'Account delivered successfully',
        ]);

        $account->decrement('stok');

        if ($account->stok <= 0) {
            $account->update(['status' => 'sold']);
        }
    }

    public function search(Request $request)
    {
        $orderId = trim((string) $request->query('order_id', ''));

        if ($orderId === '') {
            return response()->json([
                'success' => false,
                'message' => 'Nomor invoice tidak boleh kosong.',
            ], 422);
        }

        $transaction = Order::query()->where('order_id', $orderId)->first();

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Invoice tidak ditemukan.',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $transaction,
        ]);
    }

    public function searchByWhatsapp(Request $request)
    {
        $raw = (string) $request->query('whatsapp', '');
        $whatsapp = $this->normalizeWhatsapp($raw);

        if (!$whatsapp) {
            return response()->json([
                'success' => false,
                'message' => 'Nomor WhatsApp tidak boleh kosong.',
            ], 422);
        }

        $orders = Order::query()
            ->where('whatsapp', $whatsapp)
            ->orderByDesc('id')
            ->get();

        if ($orders->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Data order dengan nomor WhatsApp tersebut tidak ditemukan.',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $orders,
        ]);
    }

    public function realtimeTransaction()
    {
        $data = Order::query()
            ->orderByDesc('id')
            ->limit(10)
            ->get()
            ->map(function (Order $o) {
                return [
                    'order_id' => $o->order_id,
                    'product' => $o->product,
                    'price' => (int) ($o->price ?? 0),
                    'whatsapp' => $o->whatsapp,
                    'buy_status' => $o->buy_status,
                    'updated_at' => $o->updated_at?->toIso8601String(),
                ];
            })
            ->values();

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }
}