<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PromoCodeResource\Pages;
use App\Models\Game;
use App\Models\PaymentMethod;
use App\Models\PromoCode;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Table;
use Filament\Forms;
use Filament\Tables;

class PromoCodeResource extends Resource
{
    protected static ?string $model = PromoCode::class;

    protected static ?string $navigationGroup = 'Manajemen Game';
    protected static ?string $navigationLabel = 'Kode Promo';
    protected static ?string $pluralModelLabel = 'Kode Promo';
    protected static ?string $navigationIcon = 'heroicon-o-ticket';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Section::make('Informasi Kode Promo')
                ->schema([
                    Forms\Components\TextInput::make('name')
                        ->required()
                        ->maxLength(191),

                    Forms\Components\TextInput::make('code')
                        ->required()
                        ->maxLength(64)
                        ->dehydrateStateUsing(fn ($state) => strtoupper(trim((string) $state)))
                        ->rule('regex:/^[A-Z0-9_-]+$/')
                        ->unique(ignoreRecord: true)
                        ->validationMessages([
                            'regex' => 'Kode promo hanya boleh berisi huruf, angka, underscore, dan dash.',
                        ]),

                    Forms\Components\Toggle::make('is_active')
                        ->required()
                        ->default(true),

                    Forms\Components\DateTimePicker::make('start_at')
                        ->seconds(false),

                    Forms\Components\DateTimePicker::make('end_at')
                        ->seconds(false)
                        ->afterOrEqual('start_at'),
                ])->columns(2),

            Forms\Components\Section::make('Diskon & Syarat')
                ->schema([
                    Forms\Components\Select::make('discount_type')
                        ->required()
                        ->options([
                            'PERCENT' => 'Persen',
                            'FLAT' => 'Potongan Tetap',
                        ])
                        ->native(false),

                    Forms\Components\TextInput::make('discount_value')
                        ->required()
                        ->numeric()
                        ->minValue(1)
                        ->maxValue(fn (Forms\Get $get) => $get('discount_type') === 'PERCENT' ? 100 : null),

                    Forms\Components\TextInput::make('min_product_price')
                        ->numeric()
                        ->minValue(0)
                        ->suffix('IDR'),

                    Forms\Components\TextInput::make('usage_limit_total')
                        ->numeric()
                        ->minValue(1),

                    Forms\Components\TextInput::make('usage_limit_per_user')
                        ->numeric()
                        ->minValue(1),
                ])->columns(2),

            Forms\Components\Section::make('Batasan Opsional')
                ->schema([
                    Forms\Components\Select::make('payment_method_ids')
                        ->label('Metode Pembayaran')
                        ->multiple()
                        ->searchable()
                        ->preload()
                        ->options(fn () => PaymentMethod::query()
                            ->where('status', true)
                            ->orderBy('name')
                            ->pluck('name', 'id')
                            ->toArray()
                        )
                        ->afterStateHydrated(function (Forms\Components\Select $component, $state, $record) {
                            if (!$record) return;
                            $component->state($record->paymentMethods()->pluck('payment_methods.id')->all());
                        })
                        ->dehydrateStateUsing(fn ($state) => array_values(array_unique(array_map('intval', (array) $state))))
                        ->dehydrated(true),

                    Forms\Components\Select::make('game_ids')
                        ->label('Game')
                        ->multiple()
                        ->searchable()
                        ->preload()
                        ->options(fn () => Game::query()
                            ->orderBy('title')
                            ->pluck('title', 'id')
                            ->toArray()
                        )
                        ->afterStateHydrated(function (Forms\Components\Select $component, $state, $record) {
                            if (!$record) return;
                            $component->state($record->games()->pluck('games.id')->all());
                        })
                        ->dehydrateStateUsing(fn ($state) => array_values(array_unique(array_map('intval', (array) $state))))
                        ->dehydrated(true),
                ])->columns(2),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('code')->label('Kode')->searchable()->sortable(),
                Tables\Columns\TextColumn::make('name')->label('Nama')->searchable()->sortable(),
                Tables\Columns\IconColumn::make('is_active')->label('Aktif')->boolean()->sortable(),
                Tables\Columns\TextColumn::make('start_at')->label('Mulai')->dateTime('d M Y H:i')->sortable(),
                Tables\Columns\TextColumn::make('end_at')->label('Berakhir')->dateTime('d M Y H:i')->sortable(),
                Tables\Columns\TextColumn::make('discount_type')
                    ->label('Tipe')
                    ->formatStateUsing(fn ($state) => $state === 'PERCENT' ? 'Persen' : 'Flat')
                    ->sortable(),
                Tables\Columns\TextColumn::make('discount_value')
                    ->label('Nilai')
                    ->formatStateUsing(function ($state, PromoCode $record) {
                        if ($record->discount_type === 'PERCENT') return (int) $state . '%';
                        return 'Rp ' . number_format((int) $state, 0, ',', '.');
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('min_product_price')
                    ->label('Min Harga Produk')
                    ->formatStateUsing(fn ($state) => $state === null ? '-' : 'Rp ' . number_format((int) $state, 0, ',', '.'))
                    ->sortable(),
                Tables\Columns\TextColumn::make('used_count')->label('Dipakai')->numeric()->sortable(),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')->label('Aktif'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('id', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPromoCodes::route('/'),
            'create' => Pages\CreatePromoCode::route('/create'),
            'edit' => Pages\EditPromoCode::route('/{record}/edit'),
        ];
    }
}