<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ProductResource\Pages;
use App\Models\Product;
use App\Models\ProductCategory;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Artisan;

class ProductResource extends Resource
{
    protected static ?string $model = Product::class;

    protected static ?string $navigationGroup = 'Manajemen Produk';
    protected static ?string $navigationLabel = 'Produk';
    protected static ?string $pluralModelLabel = 'Produk';
    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static function normalizeTypeLabel(string $value): string
    {
        $v = trim($value);
        $k = mb_strtolower($v);

        if ($k === 'umum') return 'Umum';
        if ($k === 'membership') return 'Membership';

        return $v;
    }

    protected static function getTypeOptions(): array
    {
        $raw = Product::query()
            ->whereNotNull('type')
            ->where('type', '!=', '')
            ->pluck('type')
            ->map(fn ($v) => trim((string) $v))
            ->filter()
            ->values()
            ->all();

        $normalized = [];
        foreach ($raw as $t) {
            $label = static::normalizeTypeLabel($t);
            $normalized[mb_strtolower($label)] = $label;
        }

        $rest = array_values(array_filter($normalized, function ($label) {
            $k = mb_strtolower($label);
            return $k !== 'umum' && $k !== 'membership';
        }));

        usort($rest, fn ($a, $b) => strcasecmp($a, $b));

        $options = ['Umum' => 'Umum', 'Membership' => 'Membership'];
        foreach ($rest as $label) {
            $options[$label] = $label;
        }

        return $options;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\FileUpload::make('images')
                    ->image()
                    ->disk('public')
                    ->directory('product'),
                Forms\Components\FileUpload::make('banner')
                    ->image()
                    ->disk('public')
                    ->directory('product/banner'),
                Forms\Components\TextInput::make('code')
                    ->label('Product Code')
                    ->helperText('Jika produk ini adalah akun maka kode produk harus sama dengan kode akun')
                    ->unique(ignoreRecord: true)
                    ->required()
                    ->maxLength(191),
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(191),
                Forms\Components\TextInput::make('brand')
                    ->required()
                    ->helperText('Sesuaikan brand produk dengan brand pada game agar produk ini tampil sesuai dengan gamenya')
                    ->maxLength(191),
                Forms\Components\Select::make('type')
                    ->required()
                    ->options(fn () => static::getTypeOptions())
                    ->searchable()
                    ->preload(),
                Forms\Components\Select::make('category')
                    ->label('Pilih Kategori')
                    ->options(fn () => ProductCategory::pluck('title', 'id')->toArray())
                    ->searchable(),
                Forms\Components\TextInput::make('seller_name')
                    ->maxLength(191),
                Forms\Components\Select::make('type_transaction')
                    ->helperText('Transaksi manual hanya untuk jualan akun game')
                    ->options([
                        'otomatis' => 'Otomatis',
                        'manual' => 'Manual',
                    ])
                    ->required(),
                Forms\Components\TextInput::make('seller_price')
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('profit')
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('profit_gold')
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('profit_platinum')
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('selling_price')
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('selling_price_gold')
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('selling_price_platinum')
                    ->required()
                    ->numeric(),
                Forms\Components\Textarea::make('description')
                    ->required()
                    ->columnSpanFull(),
                Forms\Components\Select::make('provider')
                    ->options([
                        'Digiflazz' => 'Digiflazz',
                    ])
                    ->required(),
                Forms\Components\Toggle::make('buyer_product_status')
                    ->required(),
                Forms\Components\Toggle::make('seller_product_status')
                    ->required(),
                Forms\Components\Toggle::make('unlimited_stock')
                    ->required(),
                Forms\Components\TextInput::make('stock')
                    ->numeric()
                    ->required(),
                Forms\Components\Toggle::make('multi')
                    ->required(),
                Forms\Components\TextInput::make('start_cut_off')
                    ->maxLength(10),
                Forms\Components\TextInput::make('end_cut_off')
                    ->maxLength(10),
                Forms\Components\Toggle::make('status')
                    ->required(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('images'),
                Tables\Columns\ImageColumn::make('banner')
                    ->label('Banner'),
                Tables\Columns\TextColumn::make('code')
                    ->sortable()
                    ->searchable(),
                Tables\Columns\TextColumn::make('title')
                    ->searchable(),
                Tables\Columns\TextColumn::make('categoryRelation.title')
                    ->label('Kategori')
                    ->searchable(),
                Tables\Columns\TextColumn::make('type')
                    ->label('Type')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('seller_name')
                    ->label('Seller')
                    ->searchable(),
                Tables\Columns\TextColumn::make('start_cut_off')
                    ->label('Cutoff Mulai')
                    ->searchable(),
                Tables\Columns\TextColumn::make('end_cut_off')
                    ->label('Cutoff Selesai')
                    ->searchable(),
                Tables\Columns\TextColumn::make('seller_price')
                    ->label('Seller Price')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('selling_price')
                    ->label('Price Basic')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('selling_price_gold')
                    ->label('Price Gold')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('selling_price_platinum')
                    ->label('Price Platinum')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('profit')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('profit_gold')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('profit_platinum')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('provider')
                    ->searchable(),
                Tables\Columns\IconColumn::make('status')
                    ->boolean(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->headerActions([
                Action::make('syncProduct')
                    ->label('Sync Produk')
                    ->requiresConfirmation()
                    ->color('primary')
                    ->action(function () {
                        try {
                            Artisan::call('product:fetch-product');
                            $logFile = storage_path('logs/laravel.log');
                            $logContent = file_exists($logFile) ? file_get_contents($logFile) : '';

                            $logLines = explode("\n", trim($logContent));
                            $latestLog = end($logLines);

                            if (str_contains($latestLog, 'Failed to fetch product')) {
                                Notification::make()
                                    ->title('Error')
                                    ->body("Gagal mengambil product: " . $latestLog)
                                    ->danger()
                                    ->send();
                            } else {
                                Notification::make()
                                    ->title('Success')
                                    ->body('Product updated successfully.')
                                    ->success()
                                    ->send();
                            }
                        } catch (\Exception $e) {
                            Notification::make()
                                ->title('Error')
                                ->body('Terjadi kesalahan: ' . $e->getMessage())
                                ->danger()
                                ->send();
                        }
                    }),
            ])
            ->filters([])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getNavigationBadge(): ?string
    {
        return (string) Product::count();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProducts::route('/'),
            'create' => Pages\CreateProduct::route('/create'),
            'edit' => Pages\EditProduct::route('/{record}/edit'),
        ];
    }
}