<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PaymentMethodResource\Pages;
use App\Models\PaymentMethod;
use Filament\Forms;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Artisan;

class PaymentMethodResource extends Resource
{
    protected static ?string $model = PaymentMethod::class;

    protected static ?string $navigationGroup = 'Manajemen Website';
    protected static ?string $navigationLabel = 'Metode Pembayaran';
    protected static ?string $pluralModelLabel = 'Daftar Metode Pembayaran';

    protected static ?string $navigationIcon = 'heroicon-o-banknotes';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\FileUpload::make('images')
                    ->image()
                    ->disk('public')
                    ->directory('payment')
                    ->required(),
                Forms\Components\TextInput::make('name')
                    ->required()
                    ->maxLength(191),
                Forms\Components\Select::make('provider')
                    ->options([
                        'tripay' => 'Tripay',
                        'paydisini' => 'Paydisini',
                        'duitku' => 'Duitku',
                        'smp' => 'SMP Payment (QRIS)',
                        'qrispy' => 'QRISPY',
                    ])
                    ->required(),
                Forms\Components\TextInput::make('code')
                    ->required()
                    ->maxLength(191),
                Forms\Components\TextInput::make('group')
                    ->required()
                    ->maxLength(191),
                Forms\Components\TextInput::make('type')
                    ->required()
                    ->maxLength(191),
                Forms\Components\TextInput::make('fee')
                    ->required()
                    ->numeric()
                    ->default(0),
                Forms\Components\TextInput::make('fee_percent')
                    ->required()
                    ->numeric()
                    ->default(0),
                Forms\Components\TextInput::make('minimum_amount')
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('maximum_amount')
                    ->required()
                    ->numeric(),
                Forms\Components\Toggle::make('status')
                    ->required(),
                Forms\Components\Toggle::make('is_outside_group')
    ->label('Tampilkan di Luar Group')
    ->default(false),

Forms\Components\Select::make('badge_text')
    ->label('Badge')
    ->options([
        null => '-',
        'Best Price' => 'Best Price',
        'Harga Termurah' => 'Harga Termurah',
    ])
    ->searchable()
    ->native(false),

Forms\Components\TextInput::make('outside_sort')
    ->label('Urutan (Luar Group)')
    ->numeric()
    ->default(0),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('images')
                    ->square(),
                Tables\Columns\TextColumn::make('name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('code')
                    ->searchable(),
                Tables\Columns\TextColumn::make('provider')
                    ->searchable(),
                Tables\Columns\TextColumn::make('group')
                    ->searchable(),
                Tables\Columns\TextColumn::make('type')
                    ->searchable(),
                Tables\Columns\TextColumn::make('fee')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('fee_percent')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('minimum_amount')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('maximum_amount')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\IconColumn::make('status')
                    ->boolean(),
                Tables\Columns\IconColumn::make('is_outside_group')
    ->label('Luar Group')
    ->boolean(),

Tables\Columns\TextColumn::make('badge_text')
    ->label('Badge')
    ->searchable(),

Tables\Columns\TextColumn::make('outside_sort')
    ->label('Urutan')
    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->headerActions([
                Action::make('getPaymentChannels')
                    ->label('Get Payment Channel')
                    ->requiresConfirmation()
                    ->form([
                        Select::make('provider')
                            ->label('Pilih Provider')
                            ->options([
                                'tripay' => 'Tripay',
                            ])
                            ->default('tripay')
                            ->required(),
                    ])
                    ->color('primary')
                    ->action(function (array $data) {
                        try {
                            $provider = $data['provider'] ?? 'tripay';

                            Artisan::call('payment:fetch-channels', [
                                'provider' => $provider,
                            ]);

                            $logFile = storage_path('logs/laravel.log');
                            $logContent = file_exists($logFile) ? file_get_contents($logFile) : '';
                            $logLines = explode("\n", trim($logContent));
                            $latestLog = end($logLines);

                            if (is_string($latestLog) && str_contains($latestLog, 'Failed to fetch payment channels')) {
                                Notification::make()
                                    ->title('Error')
                                    ->body('Gagal mengambil payment channel: ' . $latestLog)
                                    ->danger()
                                    ->send();

                                return;
                            }

                            Notification::make()
                                ->title('Success')
                                ->body('Payment channels updated successfully.')
                                ->success()
                                ->send();
                        } catch (\Throwable $e) {
                            Notification::make()
                                ->title('Error')
                                ->body('Terjadi kesalahan: ' . $e->getMessage())
                                ->danger()
                                ->send();
                        }
                    }),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPaymentMethods::route('/'),
            'create' => Pages\CreatePaymentMethod::route('/create'),
            'edit' => Pages\EditPaymentMethod::route('/{record}/edit'),
        ];
    }
}