<?php

namespace App\Filament\Resources;

use App\Filament\Resources\FailedOrderResource\Pages;
use App\Models\Order;
use App\Services\DigiflazzService;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class FailedOrderResource extends Resource
{
    protected static ?string $model = Order::class;

    protected static ?string $navigationLabel = 'Failed Order';
    protected static ?string $pluralModelLabel = 'Daftar Order Failed';
    protected static ?string $navigationIcon = 'heroicon-o-exclamation-circle';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('order_id')->disabled(),
                Forms\Components\TextInput::make('games')->disabled(),
                Forms\Components\TextInput::make('product')->disabled(),
                Forms\Components\TextInput::make('code_product')->disabled(),
                Forms\Components\TextInput::make('id_games')->disabled(),
                Forms\Components\TextInput::make('server_games')->disabled(),
                Forms\Components\TextInput::make('nickname')->disabled(),
                Forms\Components\TextInput::make('email')->disabled(),
                Forms\Components\TextInput::make('whatsapp')->disabled(),
                Forms\Components\TextInput::make('code_voucher')->disabled(),
                Forms\Components\TextInput::make('price')->disabled(),
                Forms\Components\TextInput::make('discount_price')->disabled(),
                Forms\Components\TextInput::make('profit')->disabled(),
                Forms\Components\TextInput::make('total_price')->disabled(),
                Forms\Components\TextInput::make('payment_name')->disabled(),
                Forms\Components\TextInput::make('payment_method')->disabled(),
                Forms\Components\TextInput::make('payment_code')->disabled(),
                Forms\Components\TextInput::make('expired_time')->disabled(),
                Forms\Components\Select::make('payment_status')
                    ->options([
                        'PAID' => 'PAID',
                        'UNPAID' => 'UNPAID',
                        'EXPIRED' => 'EXPIRED',
                        'FAILED' => 'FAILED',
                    ])
                    ->required(),

                Forms\Components\Select::make('buy_status')
                    ->options([
                        'Sukses' => 'Sukses',
                        'Pending' => 'Pending',
                        'Gagal' => 'Gagal',
                        'Batal' => 'Batal',
                    ])
                    ->required(),

                Forms\Components\TextInput::make('serial_number')->disabled(),
                Forms\Components\Textarea::make('df_message')->disabled(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('order_id')->searchable(),
                Tables\Columns\TextColumn::make('price')->money('IDR')->sortable(),
                Tables\Columns\TextColumn::make('id_games')->searchable(),
                Tables\Columns\TextColumn::make('server_games')->searchable(),
                Tables\Columns\TextColumn::make('whatsapp')->searchable(),

                Tables\Columns\TextColumn::make('payment_status')
                    ->label('Status Pembayaran')
                    ->badge()
                    ->color(fn ($state) => match ($state) {
                        'UNPAID' => 'warning',
                        'PAID' => 'success',
                        'FAILED' => 'danger',
                        'EXPIRED' => 'danger',
                        default => 'gray',
                    }),

                Tables\Columns\TextColumn::make('buy_status')
                    ->label('Status Pembelian')
                    ->badge()
                    ->color(fn ($state) => match ($state) {
                        'Pending' => 'warning',
                        'Sukses' => 'success',
                        'Batal' => 'danger',
                        'Gagal' => 'danger',
                        default => 'gray',
                    }),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Waktu Transaksi')
                    ->dateTime()
                    ->sortable(),

                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),

                Action::make('retry')
                    ->label('Retry')
                    ->icon('heroicon-o-arrow-path')
                    ->color('warning')
                    ->requiresConfirmation()
                    ->visible(fn ($record) =>
                        $record->payment_status === 'PAID' && $record->buy_status === 'Gagal'
                    )
                    ->action(function (Order $record) {
                        $result = app(DigiflazzService::class)->processDigiflazzTransaction($record);

                        if ($result['success']) {
                            Notification::make()
                                ->title('Berhasil mengirim ulang ke Digiflazz.')
                                ->body($result['message'] ?? 'Berhasil diproses.')
                                ->success()
                                ->send();
                        } else {
                            Notification::make()
                                ->title('Gagal mengirim ulang ke Digiflazz.')
                                ->body($result['message'] ?? 'Terjadi kesalahan saat memproses ulang.')
                                ->danger()
                                ->send();
                        }
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }
    
    public static function canCreate(): bool
    {
        return false;
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListFailedOrders::route('/'),
            'create' => Pages\CreateFailedOrder::route('/create'),
            'edit' => Pages\EditFailedOrder::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->where('payment_status', 'PAID')
            ->where('buy_status', 'Gagal');
    }
    
    public static function getNavigationBadge(): ?string
    {
        return (string) \App\Models\Order::where('buy_status', 'Gagal')
            ->count();
    }
    
    public static function getNavigationBadgeColor(): ?string
    {
        return 'danger';
    }
    
}