<?php

namespace App\Filament\Pages;

use App\Models\Setting;
use App\Services\MpwaService;
use App\Services\WhatsAppNotificationService;
use Filament\Actions\Action;
use Filament\Forms\Components\Actions as FormActions;
use Filament\Forms\Components\Actions\Action as FormAction;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Radio;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Support\Exceptions\Halt;
use Illuminate\Support\HtmlString;

class NotificationSetting extends Page
{
    use InteractsWithForms;

    protected static ?string $navigationGroup = 'Manajemen Website';
    protected static ?string $navigationLabel = 'Notification Setting';
    protected static ?string $slug = 'notification-setting';
    protected static ?string $navigationIcon = 'heroicon-o-bell';
    protected static ?int $navigationSort = 20;

    protected static string $view = 'filament.pages.notification-setting';

    public array $data = [];
    public string $testPreview = '';

    public function mount(): void
    {
        $this->form->fill($this->getInitialData());
        $this->testPreview = '';
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('save')->label('Simpan')->action('save'),
        ];
    }

    public function form(Form $form): Form
    {
        return $form
            ->statePath('data')
            ->schema([
                Tabs::make()
                    ->schema([
                        Tabs\Tab::make('Provider')
                            ->schema([
                                Radio::make('notification.provider')
                                    ->options([
                                        'fonnte' => 'Fonnte',
                                        'mpwa' => 'MPWA',
                                    ])
                                    ->required()
                                    ->reactive(),

                                Section::make('Fonnte')
                                    ->visible(fn ($get) => $get('notification.provider') === 'fonnte')
                                    ->schema([
                                        TextInput::make('fonnte.token')->password()->revealable(),
                                    ]),

                                Section::make('MPWA')
                                    ->visible(fn ($get) => $get('notification.provider') === 'mpwa')
                                    ->schema([
                                        TextInput::make('mpwa.api_key')->password()->revealable(),
                                        TextInput::make('mpwa.sender')->label('Nomor Device')->helperText('Gunakan format 62xxxx'),
                                        TextInput::make('mpwa.username')->helperText('Untuk API info-user'),
                                        TextInput::make('mpwa.footer')->label('Footer Default'),

                                        FormActions::make([
                                            FormAction::make('refreshMpwaInfo')
                                                ->label('Ambil User & Device Info')
                                                ->action(fn () => $this->refreshMpwa()),
                                        ]),

                                        Section::make('User Info')
                                            ->schema([
                                                Grid::make(2)->schema([
                                                    TextInput::make('mpwa.user_info.info.id')->disabled(),
                                                    TextInput::make('mpwa.user_info.info.username')->disabled(),
                                                    TextInput::make('mpwa.user_info.info.email')->disabled(),
                                                ]),
                                            ]),

                                        Section::make('Device Info')
                                            ->schema([
                                                Repeater::make('mpwa.device_info.info')
                                                    ->disabled()
                                                    ->addable(false)
                                                    ->deletable(false)
                                                    ->reorderable(false)
                                                    ->schema([
                                                        Grid::make(3)->schema([
                                                            TextInput::make('id')->disabled(),
                                                            TextInput::make('user_id')->disabled(),
                                                            TextInput::make('body')->label('Number')->disabled(),
                                                            TextInput::make('status')->disabled(),
                                                        ]),
                                                    ]),
                                            ]),
                                    ]),
                            ]),

                        Tabs\Tab::make('Notifikasi Order')
                            ->schema([
                                Grid::make(2)->schema([
                                    $this->templateCard('waiting_payment', 'Menunggu Pembayaran'),
                                    $this->templateCard('payment_paid', 'Pembayaran Berhasil'),
                                    $this->templateCard('payment_expired', 'Pembayaran Kadaluarsa'),
                                    $this->templateCard('payment_failed', 'Pembayaran Gagal'),
                                    $this->templateCard('buy_status', 'Update Status'),
                                    $this->templateCard('delivered', 'Pesanan Terkirim'),
                                ]),

                                Section::make('Variabel Notifikasi Order')
                                    ->collapsible()
                                    ->collapsed()
                                    ->schema([
                                        Placeholder::make('order_vars_info')
                                            ->label('')
                                            ->content($this->variablesHelp()),
                                    ]),

                                $this->testSection(),
                            ]),

                        Tabs\Tab::make('Notifikasi OTP & Reset')
                            ->schema([
                                Grid::make(2)->schema([
                                    $this->templateCard('otp_register', 'OTP Register'),
                                    $this->templateCard('otp_login', 'OTP Login'),
                                    $this->templateCard('otp_reset_password', 'OTP Reset Password'),
                                    $this->templateCard('reset_password_link_whatsapp', 'Reset Password Link (WhatsApp)'),
                                ]),

                                Section::make('Variabel OTP & Reset')
                                    ->collapsible()
                                    ->collapsed()
                                    ->schema([
                                        Placeholder::make('otp_vars_info')
                                            ->label('')
                                            ->content($this->otpVariablesHelp()),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    private function templateCard(string $key, string $label): Section
    {
        return Section::make($label)
            ->collapsible()
            ->collapsed(fn () => ! data_get($this->data, "notification.enabled.$key", true))
            ->schema([
                Toggle::make("notification.enabled.$key")
                    ->label('Aktif')
                    ->default(true),

                Textarea::make("notification.templates.$key")
                    ->label('Isi Template')
                    ->rows(7),
            ]);
    }

    private function testSection(): Section
    {
        return Section::make('Test Kirim')
            ->collapsible()
            ->schema([
                Select::make('test.template_key')
                    ->options([
                        'waiting_payment' => 'Menunggu Pembayaran',
                        'payment_paid' => 'Pembayaran Berhasil',
                        'payment_expired' => 'Pembayaran Kadaluarsa',
                        'payment_failed' => 'Pembayaran Gagal',
                        'buy_status' => 'Update Status',
                        'delivered' => 'Pesanan Terkirim',
                        'otp_register' => 'OTP Register',
                        'otp_login' => 'OTP Login',
                        'otp_reset_password' => 'OTP Reset Password',
                        'reset_password_link_whatsapp' => 'Reset Password Link (WhatsApp)',
                    ])
                    ->required(),

                TextInput::make('test.phone')
                    ->label('Nomor Tujuan (opsional)')
                    ->helperText('Default akan menggunakan nomor dummy'),

                FormActions::make([
                    FormAction::make('preview')
                        ->label('Preview')
                        ->action(fn () => $this->previewTest()),

                    FormAction::make('send')
                        ->label('Kirim')
                        ->action(fn () => $this->sendTest()),
                ]),

                Placeholder::make('preview')
                    ->label('Hasil Preview')
                    ->content(fn () => $this->testPreview ? new HtmlString(nl2br(e($this->testPreview))) : '-'),
            ]);
    }

    private function previewTest(): void
    {
        $templateKey = data_get($this->data, 'test.template_key');
        if (! $templateKey) {
            throw new Halt();
        }

        $template = data_get($this->data, "notification.templates.$templateKey");
        $this->testPreview = $this->renderTemplate($template, $this->dummyVars()) ?? '';
    }

    private function sendTest(): void
    {
        $templateKey = data_get($this->data, 'test.template_key');
        if (! $templateKey) {
            throw new Halt();
        }

        $phone = data_get($this->data, 'test.phone') ?: '628123456789';

        app(WhatsAppNotificationService::class)
            ->send($phone, $templateKey, $this->dummyVars());

        Notification::make()
            ->title('Test terkirim')
            ->success()
            ->send();
    }

    private function dummyVars(): array
    {
        return [
            'app_name' => config('app.name'),
            'order_id' => 'INV-TEST',
            'games' => 'Mobile Legends',
            'product' => 'Diamond',
            'nickname' => 'Player',
            'price' => '10.000',
            'fee' => '500',
            'total_price' => '10.500',

            'purpose' => 'login',
            'otp' => '123456',
            'otp_code' => '123456',
            'expires_minutes' => 5,
            'expired_minutes' => 5,
            'whatsapp' => '628123456789',
            'email' => 'user@email.com',
            'reset_url' => 'https://topup.example.com/reset-password?token=TEST&whatsapp=628123456789',
            'date_time' => now()->format('d/m/Y H:i'),
            'ip_address' => '127.0.0.1',
            'device' => 'Mozilla/5.0',
            'support_whatsapp' => '628111111111',
        ];
    }

    private function renderTemplate(?string $template, array $vars): ?string
    {
        if (! is_string($template) || trim($template) === '') {
            return null;
        }

        foreach ($vars as $k => $v) {
            $template = str_replace('{' . $k . '}', (string) $v, $template);
        }

        return $template;
    }

    private function refreshMpwa(): void
    {
        $service = app(MpwaService::class);

        $this->data['mpwa']['user_info'] = $service->getUserInfo(
            data_get($this->data, 'mpwa.api_key'),
            data_get($this->data, 'mpwa.username')
        );

        $this->data['mpwa']['device_info'] = $service->getDeviceInfo(
            data_get($this->data, 'mpwa.api_key'),
            data_get($this->data, 'mpwa.sender')
        );

        $this->save();
    }

    public function save(): void
    {
        $state = $this->form->getState();

        foreach ([
            'notification.provider',
            'fonnte.token',
            'mpwa.api_key',
            'mpwa.sender',
            'mpwa.username',
            'mpwa.footer',
            'mpwa.user_info',
            'mpwa.device_info',

            'notification.enabled.waiting_payment',
            'notification.enabled.payment_paid',
            'notification.enabled.payment_expired',
            'notification.enabled.payment_failed',
            'notification.enabled.buy_status',
            'notification.enabled.delivered',

            'notification.enabled.otp_register',
            'notification.enabled.otp_login',
            'notification.enabled.otp_reset_password',
            'notification.enabled.reset_password_link_whatsapp',

            'notification.templates.waiting_payment',
            'notification.templates.payment_paid',
            'notification.templates.payment_expired',
            'notification.templates.payment_failed',
            'notification.templates.buy_status',
            'notification.templates.delivered',

            'notification.templates.otp_register',
            'notification.templates.otp_login',
            'notification.templates.otp_reset_password',
            'notification.templates.reset_password_link_whatsapp',
        ] as $key) {
            Setting::updateOrCreate(
                ['key' => $key],
                ['value' => data_get($state, $key)]
            );
        }

        Notification::make()
            ->title('Berhasil disimpan')
            ->success()
            ->send();
    }

    private function getInitialData(): array
    {
        $settings = Setting::pluck('value', 'key')->toArray();

        return [
            'notification' => [
                'provider' => $settings['notification.provider'] ?? 'fonnte',
                'enabled' => [
                    'waiting_payment' => $settings['notification.enabled.waiting_payment'] ?? true,
                    'payment_paid' => $settings['notification.enabled.payment_paid'] ?? true,
                    'payment_expired' => $settings['notification.enabled.payment_expired'] ?? true,
                    'payment_failed' => $settings['notification.enabled.payment_failed'] ?? true,
                    'buy_status' => $settings['notification.enabled.buy_status'] ?? true,
                    'delivered' => $settings['notification.enabled.delivered'] ?? true,

                    'otp_register' => $settings['notification.enabled.otp_register'] ?? true,
                    'otp_login' => $settings['notification.enabled.otp_login'] ?? true,
                    'otp_reset_password' => $settings['notification.enabled.otp_reset_password'] ?? true,
                    'reset_password_link_whatsapp' => $settings['notification.enabled.reset_password_link_whatsapp'] ?? true,
                ],
                'templates' => [
                    'waiting_payment' => $settings['notification.templates.waiting_payment'] ?? null,
                    'payment_paid' => $settings['notification.templates.payment_paid'] ?? null,
                    'payment_expired' => $settings['notification.templates.payment_expired'] ?? null,
                    'payment_failed' => $settings['notification.templates.payment_failed'] ?? null,
                    'buy_status' => $settings['notification.templates.buy_status'] ?? null,
                    'delivered' => $settings['notification.templates.delivered'] ?? null,

                    'otp_register' => $settings['notification.templates.otp_register'] ?? null,
                    'otp_login' => $settings['notification.templates.otp_login'] ?? null,
                    'otp_reset_password' => $settings['notification.templates.otp_reset_password'] ?? null,
                    'reset_password_link_whatsapp' => $settings['notification.templates.reset_password_link_whatsapp'] ?? null,
                ],
            ],
            'fonnte' => [
                'token' => $settings['fonnte.token'] ?? null,
            ],
            'mpwa' => [
                'api_key' => $settings['mpwa.api_key'] ?? null,
                'sender' => $settings['mpwa.sender'] ?? null,
                'username' => $settings['mpwa.username'] ?? null,
                'footer' => $settings['mpwa.footer'] ?? null,
                'user_info' => $settings['mpwa.user_info'] ?? null,
                'device_info' => $settings['mpwa.device_info'] ?? null,
            ],
            'test' => [
                'template_key' => 'waiting_payment',
                'phone' => null,
            ],
        ];
    }

    private function variablesHelp(): HtmlString
    {
        $lines = [
            '<strong>Format</strong>: gunakan <code>{nama_variabel}</code>',
            '',
            '<strong>Order</strong>',
            '<code>{order_id}</code> Invoice',
            '<code>{games}</code> Game',
            '<code>{product}</code> Produk',
            '<code>{nickname}</code> Nickname',
            '<code>{id_games}</code> ID Game',
            '<code>{server_games}</code> Server',
            '',
            '<strong>Pembayaran</strong>',
            '<code>{price}</code> Harga',
            '<code>{fee}</code> Fee',
            '<code>{total_price}</code> Total',
            '<code>{payment_name}</code> Channel',
            '<code>{payment_method}</code> Metode',
            '<code>{payment_code}</code> Kode Bayar',
            '<code>{payment_status}</code> Status',
            '<code>{expired_time}</code> Expired',
            '',
            '<strong>Status</strong>',
            '<code>{buy_status}</code> Status beli',
            '<code>{status}</code> Status provider',
            '<code>{df_message}</code> Pesan provider',
            '',
            '<strong>Delivery</strong>',
            '<code>{serial_number}</code> SN / detail',
            '',
            '<strong>System</strong>',
            '<code>{invoice_url}</code> Link invoice',
            '<code>{app_name}</code> Nama aplikasi',
        ];

        return new HtmlString(
            implode('<br>', array_map(fn ($l) => $l === '' ? '&nbsp;' : $l, $lines))
        );
    }

    private function otpVariablesHelp(): HtmlString
    {
        $lines = [
            '<strong>Format</strong>: gunakan <code>{nama_variabel}</code>',
            '',
            '<strong>OTP</strong>',
            '<code>{app_name}</code> Nama aplikasi',
            '<code>{purpose}</code> register / login / reset_password',
            '<code>{otp_code}</code> Kode OTP',
            '<code>{otp}</code> Alias OTP (kompatibel template lama)',
            '<code>{expires_minutes}</code> Masa berlaku (menit)',
            '<code>{expired_minutes}</code> Alias masa berlaku (kompatibel template lama)',
            '<code>{whatsapp}</code> Nomor tujuan',
            '<code>{email}</code> Email user (jika ada)',
            '<code>{date_time}</code> Waktu kirim',
            '',
            '<strong>Reset Password</strong>',
            '<code>{reset_url}</code> Link reset password',
            '',
            '<strong>Opsional</strong>',
            '<code>{ip_address}</code> IP',
            '<code>{device}</code> User Agent',
            '<code>{support_whatsapp}</code> WA support',
        ];

        return new HtmlString(
            implode('<br>', array_map(fn ($l) => $l === '' ? '&nbsp;' : $l, $lines))
        );
    }
}
