<?php

namespace App\Filament\Pages;

use Filament\Pages\Page;
use Illuminate\Support\Facades\File;

class LogViewer extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-document-text';
    protected static ?string $navigationLabel = 'Log Viewer';
    protected static string $view = 'filament.pages.log-viewer';
    protected static ?string $navigationGroup = 'Manajemen Website';

    public int $limit = 50;
    public string $level = 'ALL';
    public string $search = '';

    public function getLogs(): array
    {
        $logFile = storage_path('logs/laravel.log');

        if (!File::exists($logFile)) {
            return [];
        }

        $content = File::get($logFile);
        if (trim($content) === '') {
            return [];
        }

        $lines = preg_split("/\r\n|\n|\r/", $content);
        $entries = [];
        $current = [];

        for ($i = count($lines) - 1; $i >= 0; $i--) {
            $line = $lines[$i];

            if ($line === null) {
                continue;
            }

            if ($line !== '' && str_starts_with($line, '[')) {
                array_unshift($current, $line);
                $entryText = trim(implode("\n", $current));

                if ($entryText !== '') {
                    $parsed = $this->parseEntry($entryText);

                    if ($this->matchLevel($parsed) && $this->matchSearch($parsed)) {
                        $entries[] = $parsed;
                    }
                }

                $current = [];

                if (count($entries) >= $this->limit) {
                    break;
                }

                continue;
            }

            if ($line !== '') {
                array_unshift($current, $line);
            }
        }

        if (count($entries) < $this->limit && !empty($current)) {
            $entryText = trim(implode("\n", $current));
            if ($entryText !== '') {
                $parsed = $this->parseEntry($entryText);
                if ($this->matchLevel($parsed) && $this->matchSearch($parsed)) {
                    $entries[] = $parsed;
                }
            }
        }

        return $entries;
    }

    private function matchLevel(array $entry): bool
    {
        if ($this->level === 'ALL') {
            return true;
        }

        $lvl = strtoupper((string) ($entry['level'] ?? ''));
        return $lvl === strtoupper($this->level);
    }

    private function matchSearch(array $entry): bool
    {
        $q = trim($this->search);
        if ($q === '') {
            return true;
        }

        $hay = mb_strtolower(($entry['raw'] ?? '') . "\n" . ($entry['message'] ?? '') . "\n" . ($entry['details'] ?? ''));
        $needle = mb_strtolower($q);

        return str_contains($hay, $needle);
    }

    private function parseEntry(string $text): array
    {
        $firstLine = strtok($text, "\n");
        $rest = trim(substr($text, strlen($firstLine)));

        $timestamp = null;
        $env = null;
        $level = null;
        $message = $firstLine;

        if (preg_match('/^\[(.*?)\]\s+([^.]+)\.([A-Z]+):\s*(.*)$/', $firstLine, $m)) {
            $timestamp = $m[1] ?? null;
            $env = $m[2] ?? null;
            $level = $m[3] ?? null;
            $message = $m[4] ?? $message;
        }

        return [
            'timestamp' => $timestamp,
            'env' => $env,
            'level' => $level,
            'message' => $message,
            'details' => $rest,
            'raw' => $text,
        ];
    }

    public function clearLog(): void
    {
        $logFile = storage_path('logs/laravel.log');

        if (File::exists($logFile)) {
            File::put($logFile, '');
        }
    }
}